/*
OpenIO SDS metautils
Copyright (C) 2014 Worldine, original work as part of Redcurrant
Copyright (C) 2015 OpenIO, modified as part of OpenIO Software Defined Storage

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 3.0 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.
*/

#include <stddef.h>
#include <stdint.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>

#include "metautils.h"
#include "./Parameter.h"
#include "./Message.h"

enum message_param_e { MP_ID, MP_NAME, MP_VERSION, MP_BODY };

static void
_free_Parameter(Parameter_t * p)
{
	if (!p)
		return;
	ASN_STRUCT_FREE(asn_DEF_Parameter, p);
}

static OCTET_STRING_t *
__getParameter(MESSAGE m, enum message_param_e mp)
{
	switch (mp) {
		case MP_ID:
			return m->id;
		case MP_NAME:
			return m->name;
		case MP_VERSION:
			return m->version;
		case MP_BODY:
			return m->body;
		default:
			g_assert_not_reached();
			return NULL;
	}
}

MESSAGE
metautils_message_create(void)
{
	const char *id = oio_ext_get_reqid ();
	MESSAGE result = calloc(1, sizeof(Message_t));
	if (id)
		metautils_message_set_ID (result, id, strlen(id));
	return result;
}

MESSAGE
metautils_message_create_named (const char *name)
{
	MESSAGE result = metautils_message_create ();
	if (name)
		metautils_message_set_NAME (result, name, strlen(name));
	return result;
}

void
metautils_message_destroy(MESSAGE m)
{
	if (!m)
		return ;

	if (m->id != NULL)
		ASN_STRUCT_FREE(asn_DEF_OCTET_STRING, m->id);
	if (m->body != NULL)
		ASN_STRUCT_FREE(asn_DEF_OCTET_STRING, m->body);
	if (m->version != NULL)
		ASN_STRUCT_FREE(asn_DEF_OCTET_STRING, m->version);
	if (m->name != NULL)
		ASN_STRUCT_FREE(asn_DEF_OCTET_STRING, m->name);

	m->content.list.free = _free_Parameter;
	asn_set_empty(&(m->content.list));
	free(m);
}

int
metautils_asn1c_write_gba (const void *b, gsize bSize, void *key)
{
	if (b && bSize > 0)
		g_byte_array_append((GByteArray*)key, b, bSize);
	return 0;
}


GByteArray*
message_marshall_gba(MESSAGE m, GError **err)
{
	asn_enc_rval_t encRet;

	/*sanity check */
	if (!m) {
		GSETERROR(err, "Invalid parameter");
		return NULL;
	}

	/*set an ID if it is not present */
	if (!metautils_message_has_ID(m)) {
		const char *reqid = oio_ext_get_reqid ();
		if (!reqid)
			oio_ext_set_random_reqid ();
		reqid = oio_ext_get_reqid ();
		metautils_message_set_ID(m, (guint8*)reqid, strlen(reqid));
	}

	/*try to encode */
	guint32 u32 = 0;
	GByteArray *result = g_byte_array_sized_new(256);
	g_byte_array_append(result, (guint8*)&u32, sizeof(u32));
	encRet = der_encode(&asn_DEF_Message, m, metautils_asn1c_write_gba, result);

	if (encRet.encoded < 0) {
		g_byte_array_free(result, TRUE);
		GSETERROR(err, "Encoding error (Message)");
		return NULL;
	}

	guint32 s32 = result->len - 4;
	*((guint32*)(result->data)) = g_htonl(s32);
	return result;
}

GByteArray*
message_marshall_gba_and_clean(MESSAGE m)
{
	GByteArray *result;

	EXTRA_ASSERT(m != NULL);
	result = message_marshall_gba(m, NULL);
	metautils_message_destroy(m);
	return result;
}

MESSAGE
message_unmarshall(const guint8 *buf, gsize len, GError ** error)
{
	if (!buf || len < 4) {
		GSETERROR(error, "Invalid parameter");
		return NULL;
	}

	guint32 l0 = *((guint32*)buf);
	l0 = g_ntohl(l0);

	if (l0 > len-4) {
		GSETERROR(error, "l4v: uncomplete");
		return NULL;
	}

	MESSAGE m = NULL;
	asn_codec_ctx_t codec_ctx;
	codec_ctx.max_stack_size = ASN1C_MAX_STACK;
	size_t s = l0;
	asn_dec_rval_t rc = ber_decode(&codec_ctx, &asn_DEF_Message, (void**)&m, buf+4, s);

	if (rc.code == RC_OK)
		return m;

	if (rc.code == RC_WMORE)
		GSETERROR(error, "%s (%"G_GSIZE_FORMAT" bytes consumed)", "uncomplete content", rc.consumed);
	else
		GSETERROR(error, "%s (%"G_GSIZE_FORMAT" bytes consumed)", "invalid content", rc.consumed);

	metautils_message_destroy (m);
	return NULL;
}

static void*
message_get_param(MESSAGE m, enum message_param_e mp, gsize *sSize)
{
	EXTRA_ASSERT (m != NULL);

	OCTET_STRING_t *os = __getParameter(m, mp);
	if (!os || !os->buf) {
		if (sSize) *sSize = 0;
		return NULL;
	} else {
		if (sSize) *sSize = os->size;
		return os->buf;
	}
}

static void
_os_set (OCTET_STRING_t **pos, const void *s, gsize sSize)
{
	if (*pos)
		OCTET_STRING_fromBuf(*pos, s, sSize);
	else
		*pos = OCTET_STRING_new_fromBuf(&asn_DEF_OCTET_STRING, s, sSize);
}

static void
message_set_param(MESSAGE m, enum message_param_e mp, const void *s, gsize sSize)
{
	if (!m || !s || sSize < 1)
		return ;

	switch (mp) {
		case MP_ID:
			_os_set(&m->id, s, sSize);
			return;
		case MP_NAME:
			_os_set(&m->name, s, sSize);
			return;
		case MP_VERSION:
			_os_set(&m->version, s, sSize);
			return;
		case MP_BODY:
			_os_set(&m->body, s, sSize);
			return;
		default:
			g_assert_not_reached();
			return;
	}
}

void*
metautils_message_get_ID (MESSAGE m, gsize *l)
{ return message_get_param(m, MP_ID, l); }

void*
metautils_message_get_NAME (MESSAGE m, gsize *l)
{ return message_get_param(m, MP_NAME, l); }

void*
metautils_message_get_VERSION (MESSAGE m, gsize *l)
{ return message_get_param(m, MP_VERSION, l); }

void*
metautils_message_get_BODY (MESSAGE m, gsize *l)
{ return message_get_param(m, MP_BODY, l); }

void
metautils_message_set_ID (MESSAGE m, const void *b, gsize l)
{ return message_set_param(m, MP_ID, b, l); }

void
metautils_message_set_NAME (MESSAGE m, const void *b, gsize l)
{ return message_set_param(m, MP_NAME, b, l); }

void
metautils_message_set_VERSION (MESSAGE m, const void *b, gsize l)
{ return message_set_param(m, MP_VERSION, b, l); }

void
metautils_message_set_BODY (MESSAGE m, const void *b, gsize l)
{ return message_set_param(m, MP_BODY, b, l); }

gboolean
metautils_message_has_ID (MESSAGE m)
{ return NULL != metautils_message_get_ID(m,NULL); }

gboolean
metautils_message_has_NAME (MESSAGE m)
{ return NULL != metautils_message_get_NAME(m,NULL); }

gboolean
metautils_message_has_VERSION (MESSAGE m)
{ return NULL != metautils_message_get_VERSION(m,NULL); }

gboolean
metautils_message_has_BODY (MESSAGE m)
{ return NULL != metautils_message_get_BODY(m,NULL); }

void*
metautils_message_get_field(MESSAGE m, const char *name, gsize *vsize)
{
	EXTRA_ASSERT (m != NULL);
	EXTRA_ASSERT (name != NULL);
	EXTRA_ASSERT (vsize != NULL);

	*vsize = 0;

	if (!m->content.list.array) {
		return NULL;
	}

	gssize nlen = strlen(name);

	for (int i = 0; i < m->content.list.count ;i++) {
		Parameter_t *p = m->content.list.array[i];
		if (!p)
			continue;
		if (p->name.size != nlen)
			continue;

		if (!memcmp(p->name.buf, name, nlen)) {
			*vsize = p->value.size;
			return p->value.buf;
		}
	}

	return NULL;
}

gchar **
metautils_message_get_field_names(MESSAGE m)
{
	EXTRA_ASSERT(m != NULL);

	int i, nb, max = m->content.list.count;
	if (max < 0)
		return NULL;

	gchar **array = g_malloc0(sizeof(gchar *) * (max + 1));
	for (nb=0,i=0; i<max; ) {
		Parameter_t *p = m->content.list.array[i++];
		if (p && p->name.buf)
			array[nb++] = g_strndup((const gchar*)p->name.buf, p->name.size);
	}

	return array;
}

GHashTable*
metautils_message_get_fields (MESSAGE m)
{
	EXTRA_ASSERT(m != NULL);

	GHashTable *hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, metautils_gba_clean);

	for (int i=0,max=m->content.list.count; i<max; i++) {
		Parameter_t *p = m->content.list.array[i];
		if (p && p->name.buf && p->value.buf) {
			g_hash_table_replace (hash,
					g_strndup ((const gchar*)p->name.buf, p->name.size),
					g_byte_array_append (g_byte_array_new(), p->value.buf, p->value.size));
		}
	}

	return hash;
}

void
metautils_message_add_field(MESSAGE m, const char *n, const void *v, gsize vs)
{
	EXTRA_ASSERT (m!=NULL);
	EXTRA_ASSERT (n!=NULL);
	if (!v || !vs)
		return ;
	Parameter_t *pMember = calloc(1, sizeof(Parameter_t));
	OCTET_STRING_fromBuf(&(pMember->name), n, strlen(n));
	OCTET_STRING_fromBuf(&(pMember->value), v, vs);
	asn_set_add(&(m->content.list), pMember);
}

void
metautils_message_add_field_str(MESSAGE m, const char *name, const char *value)
{
	if (value)
		metautils_message_add_field (m, name, value, strlen(value));
}

void
metautils_message_add_field_gba(MESSAGE m, const char *name, GByteArray *gba)
{
	if (gba)
		metautils_message_add_field (m, name, gba->data, gba->len);
}

void
metautils_message_add_field_strint64(MESSAGE m, const char *name, gint64 v)
{
	gchar tmp[24];
	g_snprintf(tmp, 24, "%"G_GINT64_FORMAT, v);
	return metautils_message_add_field_str(m, name, tmp);
}

static struct map_s
{
	const char *f;
	int u;
	const char *avoid;
} url2msg_map[] = {
	{NAME_MSGKEY_NAMESPACE,   OIOURL_NS,        NULL},
	{NAME_MSGKEY_ACCOUNT,     OIOURL_ACCOUNT,   NULL},
	{NAME_MSGKEY_USER,        OIOURL_USER,      NULL},
	{NAME_MSGKEY_TYPENAME,    OIOURL_TYPE,      OIOURL_DEFAULT_TYPE},
	{NAME_MSGKEY_CONTENTPATH, OIOURL_PATH,      NULL},
	{NAME_MSGKEY_CONTENTID,   OIOURL_CONTENTID, NULL},
	{NAME_MSGKEY_VERSION,     OIOURL_VERSION,   NULL},
	{NULL,0,NULL},
};

void
metautils_message_add_url (MESSAGE m, struct oio_url_s *url)
{
	if (!m)
		return;
	for (struct map_s *p = url2msg_map; p->f ;++p) {
		if (oio_url_has (url, p->u)) {
			const char *s = oio_url_get (url, p->u);
			if (!p->avoid || strcmp(p->avoid, s))
				metautils_message_add_field_str(m, p->f, s);
		}
	}

	const guint8 *id = oio_url_get_id (url);
	if (id)
		metautils_message_add_field (m, NAME_MSGKEY_CONTAINERID, id, oio_url_get_id_size (url));
}

void
metautils_message_add_url_no_type (MESSAGE m, struct oio_url_s *url)
{
	if (!m)
		return;
	for (struct map_s *p = url2msg_map; p->f ;++p) {
		if (p->u != OIOURL_TYPE && oio_url_has (url, p->u)) {
			const char *s = oio_url_get (url, p->u);
			if (!p->avoid || strcmp(p->avoid, s))
				metautils_message_add_field_str(m, p->f, s);
		}
	}

	const guint8 *id = oio_url_get_id (url);
	if (id)
		metautils_message_add_field (m, NAME_MSGKEY_CONTAINERID, id, oio_url_get_id_size (url));
}

struct oio_url_s *
metautils_message_extract_url (MESSAGE m)
{
	struct oio_url_s *url = oio_url_empty ();
	for (struct map_s *p = url2msg_map; p->f ;++p) {
		// TODO call really often, so make it zero-copy
		gchar *s = metautils_message_extract_string_copy (m, p->f);
		if (s) {
			if (!p->avoid || strcmp(p->avoid, s))
				oio_url_set (url, p->u, s);
			g_free0 (s);
		}
	}

	container_id_t cid;
	GError *e = metautils_message_extract_cid (m, NAME_MSGKEY_CONTAINERID, &cid);
	if (e)
		g_clear_error (&e);
	else
		oio_url_set_id (url, cid);

	return url;
}

void
metautils_message_add_cid (MESSAGE m, const char *f, const container_id_t cid)
{
	if (cid)
		metautils_message_add_field (m, f, cid, sizeof(container_id_t));
}

void
metautils_message_add_body_unref (MESSAGE m, GByteArray *body)
{
	if (!body)
		return;
	if (body->len && body->data)
		metautils_message_set_BODY (m, body->data, body->len);
	g_byte_array_unref (body);
}

GError *
metautils_message_extract_body_strv(MESSAGE msg, gchar ***result)
{
	gsize bsize = 0;
	void *b = metautils_message_get_BODY(msg, &bsize);
	if (!b) {
		*result = g_malloc0(sizeof(gchar*));
		return NULL;
	}

	*result = metautils_decode_lines(b, ((gchar*)b)+bsize);
	return NULL;
}

GError *
metautils_message_extract_prefix(MESSAGE msg, const gchar *n,
		guint8 *d, gsize *dsize)
{
	gsize fsize = 0;
	void *f = metautils_message_get_field(msg, n, &fsize);
	if (!f || fsize)
		return NEWERROR(CODE_BAD_REQUEST, "Missing ID prefix at '%s'", n);
	if (fsize > *dsize)
		return NEWERROR(CODE_BAD_REQUEST, "Invalid ID prefix at '%s'", n);

	memset(d, 0, *dsize);
	memcpy(d, f, fsize);
	*dsize = fsize;
	return NULL;
}

GError *
metautils_message_extract_cid(MESSAGE msg, const gchar *n, container_id_t *cid)
{
	gsize fsize = 0;
	void *f = metautils_message_get_field(msg, n, &fsize);
	if (!f || !fsize)
		return NEWERROR(CODE_BAD_REQUEST, "Missing container ID at '%s'", n);
	if (fsize != sizeof(container_id_t))
		return NEWERROR(CODE_BAD_REQUEST, "Invalid container ID at '%s'", n);
	memcpy(cid, f, sizeof(container_id_t));
	return NULL;
}

GError *
metautils_message_extract_string(MESSAGE msg, const gchar *n, gchar *dst,
		gsize dst_size)
{
	gsize fsize = 0;
	void *f = metautils_message_get_field(msg, n, &fsize);
	if (!f)
		return NEWERROR(CODE_BAD_REQUEST, "missing '%s'", n);
	if (!fsize)
		return NEWERROR(CODE_BAD_REQUEST, "empty '%s'", n);
	if ((gssize)fsize < 0 || fsize >= dst_size)
		return NEWERROR(CODE_BAD_REQUEST, "too long '%s' (%"G_GSIZE_FORMAT")", n, fsize);

	if (fsize)
		memcpy(dst, f, fsize);
	memset(dst+fsize, 0, dst_size-fsize);
	return NULL;
}

gchar *
metautils_message_extract_string_copy(MESSAGE msg, const gchar *n)
{
	gsize fsize = 0;
	void *f = metautils_message_get_field(msg, n, &fsize);
	if (!f || !fsize)
		return NULL;
	return g_strndup(f, fsize);
}

gboolean
metautils_message_extract_flag(MESSAGE msg, const gchar *n, gboolean def)
{
	gsize fsize = 0;
	void *f = metautils_message_get_field(msg, n, &fsize);
	if (!f || !fsize)
		return def;

	guint8 *b, _flag = 0;
	for (b=(guint8*)f + fsize; b > (guint8*)f;)
		_flag |= *(--b);
	return _flag;
}

GError*
metautils_message_extract_flags32(MESSAGE msg, const gchar *n,
		gboolean mandatory, guint32 *flags)
{
	EXTRA_ASSERT(flags != NULL);
	*flags = 0;

	gsize fsize = 0;
	void *f = metautils_message_get_field(msg, n, &fsize);
	if (!f || !fsize) {
		if (mandatory)
			return NEWERROR(CODE_BAD_REQUEST, "Missing field '%s'", n);
		return NULL;
	}

	if (fsize != 4)
		return NEWERROR(CODE_BAD_REQUEST, "Invalid 32bit flag set");

	*flags = g_ntohl(*((guint32*)f));
	return NULL;
}

GError *
metautils_message_extract_body_gba(MESSAGE msg, GByteArray **result)
{
	EXTRA_ASSERT(result != NULL);

	gsize bsize = 0;
	*result = NULL;
	void *b = metautils_message_get_BODY(msg, &bsize);
	if (!b)
		return NEWERROR(CODE_BAD_REQUEST, "No body");

	*result = g_byte_array_new();
	if (b && bsize)
		g_byte_array_append(*result, b, bsize);
	return NULL;
}

GError *
metautils_message_extract_body_string(MESSAGE msg, gchar **result)
{
	gsize bsize = 0;
	void *b = metautils_message_get_BODY(msg, &bsize);
	if (!b)
		return NEWERROR(CODE_BAD_REQUEST, "No body");

	if (!b || !bsize) {
		*result = g_malloc0(sizeof(void*));
		return NULL;
	}

	register gchar *c, *last;
	for (c=b,last=b+bsize; c < last ;c++) {
		if (!g_ascii_isprint(*c))
			return NEWERROR(CODE_BAD_REQUEST,
					"Body contains non printable characters");
	}

	*result = g_strndup((gchar*)b, bsize);
	return NULL;
}

GError *
metautils_unpack_bodyv (GByteArray **bodyv, GSList **result,
		body_decoder_f decoder)
{
	GError *err = NULL;
	GSList *items = NULL;
	for (GByteArray **p=bodyv; *p && !err ;++p) {
		GSList *l = NULL;
		if (!decoder (&l, (*p)->data, (*p)->len, NULL))
			err = NEWERROR (CODE_PROXY_ERROR, "Bad payload from service");
		else
			items = metautils_gslist_precat (items, l);
	}
	*result = items;
	return err;
}

GError *
metautils_message_extract_body_encoded(MESSAGE msg, gboolean mandatory,
		GSList **result, body_decoder_f decoder)
{
	EXTRA_ASSERT(result != NULL);
	EXTRA_ASSERT(decoder != NULL);

	gsize bsize = 0;
	void *b = metautils_message_get_BODY(msg, &bsize);
	if (!b) {
		if (mandatory)
			return NEWERROR(CODE_BAD_REQUEST, "Missing body");
		return NULL;
    }

	GError *err = NULL;
	int rc = decoder(result, b, bsize, &err);
	if (rc <= 0) {
		EXTRA_ASSERT(err != NULL);
		err->code = CODE_BAD_REQUEST;
		g_prefix_error(&err, "Invalid body: ");
		return err;
	}

	return NULL;
}

GError*
metautils_message_extract_header_encoded(MESSAGE msg, const gchar *n, gboolean mandatory,
		GSList **result, body_decoder_f decoder)
{
	EXTRA_ASSERT(result != NULL);
	EXTRA_ASSERT(decoder != NULL);

	gsize bsize = 0;
	void *b = metautils_message_get_field(msg, n, &bsize);
	if (!b || !bsize) {
		*result = NULL;
		if (mandatory)
			return NEWERROR(CODE_BAD_REQUEST, "Missing header [%s]", n);
		return NULL;
	}

	GError *err = NULL;
	int rc = decoder(result, b, bsize, &err);
	if (rc <= 0) {
		EXTRA_ASSERT(err != NULL);
		err->code = CODE_BAD_REQUEST;
		g_prefix_error(&err, "Invalid header: ");
		return err;
	}

	return NULL;
}

GError *
metautils_message_extract_strint64(MESSAGE msg, const gchar *n, gint64 *i64)
{
	gchar *end, dst[24];

	EXTRA_ASSERT (i64 != NULL);
	*i64 = 0;

	memset(dst, 0, sizeof(dst));
	GError *err = metautils_message_extract_string(msg, n, dst, sizeof(dst));
	if (err != NULL) {
		g_prefix_error(&err, "field: ");
		return err;
	}

	end = NULL;
	*i64 = g_ascii_strtoll(dst, &end, 10);

	switch (*i64) {
		case G_MININT64:
		case G_MAXINT64:
			return (errno == ERANGE)
				? NEWERROR(CODE_BAD_REQUEST, "Invalid number") : NULL;
		case 0:
			return (end == dst)
				? NEWERROR(CODE_BAD_REQUEST, "Invalid number") : NULL;
		default:
			return NULL;
	}
}

GError*
metautils_message_extract_struint(MESSAGE msg, const gchar *n, guint *u)
{
	EXTRA_ASSERT (u != NULL);
	*u = 0;
	gint64 i64 = 0;
	GError *err = metautils_message_extract_strint64(msg, n, &i64);
	if (err) { g_prefix_error(&err, "struint: "); return err; }
	if (i64<0) return NEWERROR(CODE_BAD_REQUEST, "[%s] is negative", n);
	if (i64 > G_MAXUINT) return NEWERROR(CODE_BAD_REQUEST, "[%s] is too big", n);
	*u = i64;
	return NULL;
}

GError*
metautils_message_extract_boolean(MESSAGE msg, const gchar *n,
		gboolean mandatory, gboolean *v)
{
	gchar tmp[16];
	GError *err = metautils_message_extract_string(msg, n, tmp, sizeof(tmp));
	if (err) {
		if (!mandatory)
			g_clear_error(&err);
		return err;
	}
	if (v)
		*v = metautils_cfg_get_bool (tmp, *v);
	return NULL;
}

