/*
 *
 * Copyright 2015, OpenIO
 *
 */

#ifndef INCLUDE_OIOFS_OIOFS_H_
#define INCLUDE_OIOFS_OIOFS_H_

#include <utime.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/statvfs.h>
#include <sys/socket.h>
#include <stdbool.h>
#include <dirent.h>


// FreeBSD and OSX Compatiblity
#if defined(__FreeBSD__) || defined(__APPLE__)
typedef off_t loff_t;
typedef off_t off64_t;
#endif

#ifdef __cplusplus
extern "C" {
#endif


struct oiofs_mount_info;
struct oiofs_dir_result;

struct FileHandle;

struct Inode;

/**
 * Create a mount info to interact with oiofs.
 *
 * @param mount_info the mount info to initialize.
 * @returns 0 on success, negative error code on failure
 */
int oiofs_create(struct oiofs_mount_info **mount_info);

/**
 * Initialize the filesystem.
 *
 * @param mount_info
 * @returns 0 on success, negative error code on failure
 */
int oiofs_init(struct oiofs_mount_info *mount_info);

/**
 * Mount the filesystem.
 *
 * @param mount_info
 * @returns 0 on success, negative error code on failure
 */
int oiofs_mount(struct oiofs_mount_info *mount_info);

/**
 * Unmount the filesystem.
 *
 * @param mount_info
 * @returns 0 on success, negative error code on failure
 */
int oiofs_unmount(struct oiofs_mount_info *mount_info);

/**
 * Release the mount_info.
 *
 * @param mount_info
 * @returns 0 on success, negative error code on failure
 */
int oiofs_release(struct oiofs_mount_info *mount_info);

/**
 * Statfs on the filesystem.
 *
 * @param mount_info
 * @param path
 * @param stat_buf filesystem stats
 * @returns 0 on success, negative error code on failure
 */
int oiofs_statfs(struct oiofs_mount_info *mount_info, const char *path, struct statvfs *stat_buf);

/**
 * Sync all data to persistent storage.
 *
 * @param mount_info
 * @returns 0 on success, negative error code on failure
 */
int oiofs_sync_fs(struct oiofs_mount_info *mount_info);

/**
 * Open the given directory.
 *
 * @param mount_info
 * @param path
 * @param dir_result
 * @returns 0 on success, negative error code on failure
 */
int oiofs_opendir(struct oiofs_mount_info *mount_info, const char *path,
    struct oiofs_dir_result **dir_result);

/**
 * Close the given open directory.
 *
 * @param mount_info
 * @param dir_result
 * @returns 0 on success, negative error code on failure
 */
int oiofs_closedir(struct oiofs_mount_info *mount_info, struct oiofs_dir_result *dir_result);

/**
 * Get the next entry in a directory.
 *
 * @param mount_info
 * @param dir_result
 * @param dir_entry
 * @returns 0 on success, negative error code on failure
 */
int oiofs_readdir(struct oiofs_mount_info *mount_info, struct oiofs_dir_result *dir_result,
    struct dirent *dir_entry);

/**
 * Create a directory.
 *
 * @param mount_info
 * @param path
 * @param mode
 * @returns 0 on success, negative error code on failure
 */
int oiofs_mkdir(struct oiofs_mount_info *mount_info, const char *path, mode_t mode);

/**
 * Remove a directory.
 *
 * @param mount_info
 * @param path
 * @returns 0 on success, negative error code on failure
 */
int oiofs_rmdir(struct oiofs_mount_info *mount_info, const char *path);

/**
 * Create a link.
 *
 * @param mount_info
 * @param existing
 * @param newname
 * @returns 0 on success, negative error code on failure
 */
int oiofs_link(struct oiofs_mount_info *mount_info, const char *existing, const char *newname);

/**
 * Read a symbolic link.
 *
 * @param mount_info
 * @param path
 * @param buf
 * @param size
 * @returns number of bytes copied on success, negative error code on failure
 */
int oiofs_readlink(struct oiofs_mount_info *mount_info, const char *path, char *buf, int64_t size);

/**
 * Create a symbolic link.
 *
 * @param mount_info
 * @param existing
 * @param newname
 * @returns 0 on success, negative error code on failure
 */
int oiofs_symlink(struct oiofs_mount_info *mount_info, const char *existing, const char *newname);

/**
 * Remove a file, link or symbolic link.
 *
 * @param mount_info
 * @param path
 * @returns 0 on success, negative error code on failure
 */
int oiofs_unlink(struct oiofs_mount_info *mount_info, const char *path);

/**
 * Rename a file or directory.
 *
 * @param mount_info
 * @param from
 * @param to
 * @returns 0 on success, negative error code on failure
 */
int oiofs_rename(struct oiofs_mount_info *mount_info, const char *from, const char *to);

/**
 * Get a file's stats.
 *
 * @param mount_info
 * @param path
 * @param stat_buf
 * @returns 0 on success, negative error code on failure
 */
int oiofs_stat(struct oiofs_mount_info *mount_info, const char *path, struct stat *stat_buf);

/**
 * Set a file's attributes.
 *
 * @param mount_info
 * @param path
 * @param attr
 * @param mask
 * @returns 0 on success, negative error code on failure
 */
int oiofs_setattr(struct oiofs_mount_info *mount_info, const char *path, struct stat *attr,
    int mask);

/**
 * Change the permissions of a file/directory.
 *
 * @param mount_info
 * @param path
 * @param mode
 * @returns 0 on success, negative error code on failure
 */
int oiofs_chmod(struct oiofs_mount_info *mount_info, const char *path, mode_t mode);

/**
 * Change the permissions of an open file.
 *
 * @param mount_info
 * @param fd
 * @param mode
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fchmod(struct oiofs_mount_info *mount_info, int fd, mode_t mode);

/**
 * Change the ownership of a file/directory.
 *
 * @param mount_info
 * @param path
 * @param uid
 * @param gid
 * @returns 0 on success, negative error code on failure
 */
int oiofs_chown(struct oiofs_mount_info *mount_info, const char *path, int uid, int gid);

/**
 * Change the ownership of a open file.
 *
 * @param mount_info
 * @param fd
 * @param uid
 * @param gid
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fchown(struct oiofs_mount_info *mount_info, const char *path, int uid, int gid);

/**
 * Change a file/directory last access and modifications times.
 *
 * @param mount_info
 * @param path
 * @param buf
 * @returns 0 on success, negative error code on failure
 */
int oiofs_utime(struct oiofs_mount_info *mount_info, const char *path, struct utimbuf *buf);

/**
 * Truncate a file to the given size.
 *
 * @param mount_info
 * @param path
 * @param size
 * @returns 0 on success, negative error code on failure
 */
int oiofs_truncate(struct oiofs_mount_info *mount_info, const char *path, int64_t size);

/**
 * Truncate an open file to the given size.
 *
 * @param mount_info
 * @param fd
 * @param size
 * @returns 0 on success, negative error code on failure
 */
int oiofs_ftruncate(struct oiofs_mount_info *mount_info, int fd, int64_t size);

/**
 * Create a block or character special file.
 *
 * @param mount_info
 * @param path
 * @param mode
 * @param rdev
 * @returns 0 on success, negative error code on failure
 */
int oiofs_mknod(struct oiofs_mount_info *mount_info, const char *path, mode_t mode, dev_t rdev);

/**
 * Create and/or open a file.
 *
 * @param mount_info
 * @param path
 * @param flags
 * @param mode
 * @param out
 * @returns a file descriptor number on success, negative error code on failure
 */
int oiofs_open(struct oiofs_mount_info *mount_info, const char *path, int flags, mode_t mode,
    FileHandle **out);

/**
 * Close the open file.
 *
 * @param mount_info
 * @param fd
 * @returns 0 on success, negative error code on failure
 */
int oiofs_close(struct oiofs_mount_info *mount_info, int fd);

/**
 * Seek the open file stream to the given offset.
 *
 * @param mount_info
 * @param fd
 * @param offset
 * @param whence
 * @returns 0 on success, negative error code on failure
 */
int oiofs_lseek(struct oiofs_mount_info *mount_info, int fd, int64_t offset, int whence);

/**
 * Read data from the open file.
 *
 * @param mount_info
 * @param fd
 * @param buf
 * @param size
 * @param offset
 * @returns number of bytes read into buf on success, negative error code on failure
 */
int oiofs_read(struct oiofs_mount_info *mount_info, int fd, char *buf, int64_t size,
    int64_t offset);

/**
 * Write data to the open file.
 *
 * @param mount_info
 * @param fd
 * @param buf
 * @param size
 * @param offset
 * @returns number of bytes written on success, negative error code on failure
 */
int oiofs_write(struct oiofs_mount_info *mount_info, int fd, const char *buf, int64_t size,
    int64_t offset);

/**
 * Synchronize an open file to persistent storage.
 *
 * @param mount_info
 * @param fd
 * @param syncdataonly
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fsync(struct oiofs_mount_info *mount_info, int fd, int syncdataonly);

/**
 * Preallocate or release disk space for the file for the given byte range.
 *
 * @param mount_info
 * @param fd
 * @param mode
 * @param offset
 * @param length
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fallocate(struct oiofs_mount_info *mount_info, int fd, int mode, int64_t offset,
    int64_t length);

/**
 * Get the open file's stats.
 *
 * @param mount_info
 * @param fd
 * @param stat_buf
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fstat(struct oiofs_mount_info *mount_info, int fd, struct stat *stat_buf);

/**
 * Get a file's extended attribute.
 *
 * @param mount_info
 * @param path
 * @param name
 * @param value
 * @param size
 * @returns size of the value on success, negative error code on failure
 */
int oiofs_getxattr(struct oiofs_mount_info *mount_info, const char *path, const char *name,
    void *value, size_t size);

/**
 * Get an open file's extended attribute.
 *
 * @param mount_info
 * @param fd
 * @param name
 * @param value
 * @param size
 * @returns size of the value on success, negative error code on failure
 */
int oiofs_fgetxattr(struct oiofs_mount_info *mount_info, int fd, const char *name, void *value,
    size_t size);

/**
 * List the extended attribute keys on a file.
 *
 * @param mount_info
 * @param path
 * @param list
 * @param size
 * @returns size of the list on success, negative error code on failure
 */
int oiofs_listxattr(struct oiofs_mount_info *mount_info, const char *path, char *list, size_t size);

/**
 * List the extended attribute keys on a open file.
 *
 * @param mount_info
 * @param fd
 * @param list
 * @param size
 * @returns size of the list on success, negative error code on failure
 */
int oiofs_flistxattr(struct oiofs_mount_info *mount_info, int fd, char *list, size_t size);

/**
 * Remove an extended attribute from a file.
 *
 * @param mount_info
 * @param path
 * @param name
 * @returns 0 on success, negative error code on failure
 */
int oiofs_removexattr(struct oiofs_mount_info *mount_info, const char *path, const char *name);

/**
 * Remove an extended attribute from a open file.
 *
 * @param mount_info
 * @param fd
 * @param name
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fremovexattr(struct oiofs_mount_info *mount_info, int fd, const char *name);

/**
 * Set an extended attribute on a file.
 *
 * @param mount_info
 * @param path
 * @param name
 * @param value
 * @param size
 * @param flags
 * @returns 0 on success, negative error code on failure
 */
int oiofs_setxattr(struct oiofs_mount_info *mount_info, const char *path, const char *name,
    const void *value, size_t size, int flags);

/**
 * Set an extended attribute on a open file.
 *
 * @param mount_info
 * @param fd
 * @param name
 * @param value
 * @param size
 * @param flags
 * @returns 0 on success, negative error code on failure
 */
int oiofs_fsetxattr(struct oiofs_mount_info *mount_info, int fd, const char *name,
    const void *value, size_t size, int flags);

#ifdef __cplusplus
}
#endif

#endif  // INCLUDE_OIOFS_OIOFS_H_
