/*
 *
 * Copyright 2015, OpenIO
 *
 */

#include <fcntl.h>
#include <glog/logging.h>
#include "include/oiofs.h"
#include "src/page_list.h"
#include "src/file_cache.h"
#include <sys/stat.h>
#include <assert.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <algorithm>
#include <cstdio>
#include <sstream>
#include <iostream>
#include <vector>

#ifndef FLOG
# ifdef __GNUC__
#  define FLOG(level) DVLOG(level) << __PRETTY_FUNCTION__
# else
#  define FLOG(level) DVLOG(level) << __FUNCTION__
# endif
#endif


FileHandle::FileHandle(struct oio_sds_s *oio, struct oio_url_s *oio_url, const char *path)
  : pagelist(nullptr), inode(nullptr), ref(0), mode(0), fd(-1), pfile(nullptr), refcnt(0),
    is_modified(false), oio_client(oio), url(oio_url), final_size(0) {
  cache_path = strdup(path);
}

FileHandle::~FileHandle() {
  Clear();
  if (cache_path != nullptr) {
    free(cache_path);
    cache_path = nullptr;
  }
  oio_url_pclean(&url);
}

int FileHandle::FillFile(int fd, unsigned char byte, size_t size, off_t start) {
  unsigned char bytes[1024 * 64];
  memset(bytes, byte, std::min(sizeof(bytes), size));

  for (ssize_t total = 0, onewrote = 0; static_cast<size_t>(total) < size; total += onewrote) {
    if (-1 == (onewrote = pwrite(fd, bytes, std::min(sizeof(bytes),
              (size - static_cast<size_t>(total))), start + total))) {
      return -errno;
    }
  }
  return 0;
}

void FileHandle::Clear() {
  if (pfile) {
    fclose(pfile);
    pfile = nullptr;
    fd = -1;
  }
  refcnt = 0;
}

int FileHandle::Dup() {
  if (-1 != fd) {
    refcnt++;
  }
  return fd;
}

void FileHandle::Close() {
  if (-1 != fd) {
    if (0 < refcnt) {
      refcnt--;
    }
    if (0 == refcnt) {
      if (pfile != nullptr) {
        fclose(pfile);
        pfile = nullptr;
        fd = -1;
      } else {
        close(fd);
      }
    }
  }
}

int FileHandle::Open() {
  FLOG(1) << ", ino=" << inode->ino << " fd=" << fd;

  if (-1 != fd) {
    Dup();
    return 0;
  }

  // The `mode` must always allow us to read and write
  if (-1 == (fd = open(cache_path, O_CREAT|O_RDWR, 0600)) ||
      (nullptr == (pfile = fdopen(fd, "r+")))) {
    LOG(ERROR) << "failed to open cache file [" << cache_path <<"]: "
        << strerror(errno);
    int prev_errno = errno;
    if (fd >= 0) {
      close(fd);
      fd = -1;
    }
    return (0 == prev_errno ? -EIO : -prev_errno);
  }

  if (!inode->dirty) {
    pagelist->Resize(inode->size, false);
    FLOG(2) << " ftruncate(fd=" << fd << ", " << inode->size << ")";
    if (0 != ftruncate(fd, static_cast<off_t>(inode->size)) || 0 != fsync(fd)) {
      // TODO(noone) error ftruncate failed
      LOG(ERROR) << "ftruncate/fsync failed (" << cache_path << "): "
          << strerror(errno);
      fclose(pfile);
      pfile = nullptr;
      fd = -1;
      return (0 == errno ? -EIO : -errno);
    }
  }

  refcnt = 1;
  is_modified = false;
  return 0;
}

int FileHandle::EnsureOpen() {
  FLOG(1) << " ino=" << inode->ino << ", fd=" << fd;
  if (!IsOpen()) {
    int err = Open();
    if (err) {
      return err;
    }
  }
  if (-1 == fd) {
    return -EBADF;
  }
  return 0;
}

ssize_t FileHandle::Read(char *buf, off_t start, size_t size, bool force_load) {
  FLOG(1) << " fd=" << fd << ", offset=" << start << ", size=" << size
      << " forced=" << force_load;
  int err = EnsureOpen();
  if (err)
    return err;

  if (force_load) {
    pagelist->SetPageLoadedStatus(start, size, false);
  }

  int result;
  ssize_t rsize;

  if (0 < pagelist->GetTotalUnloadedPageSize(start, size)) {
    if (!FileCache::IsSafeDiskSpace(size)) {
      return -ENOSPC;
    }

    // Prefetch size
    size_t load_size = size;
    if (static_cast<size_t>(start + size) < pagelist->Size()) {
      // TODO(noone) find prefetch size
      size_t max_prefetch_size = 1024*1024*10;
      size_t prefetch_size = std::max(max_prefetch_size, size);

      if (static_cast<size_t>(start + prefetch_size) < pagelist->Size()) {
        load_size = prefetch_size;
      } else {
        load_size = static_cast<size_t>(pagelist->Size() - start);
      }
    }

    // Load data
    if ((0 < size) && 0 != (result = Load(start, load_size))) {
      // TODO(noone) error could not load data
      return -EIO;
    }
  }

  if (-1 == (rsize = pread(fd, buf, size, start))) {
    // TODO(noone) error log unable to read file
    return -errno;
  }
  return rsize;
}

struct brun_dl {
  int fd;
};

static int _write_FILE(void *ctx, const unsigned char *buf, size_t len) {
  brun_dl *b = static_cast<brun_dl*>(ctx);
  size_t total_write = 0;
  while (total_write < len) {
    ssize_t write_bytes = 0;
    write_bytes = write(b->fd, buf+total_write, len - total_write);
    if (0 == write_bytes) {
      break;
    } else if (0 > write_bytes) {
      // TODO(noone) error write file
      return -1;
    }
    total_write += static_cast<size_t>(write_bytes);
  }
  return 0;
}

int FileHandle::Load(off_t start, size_t size) {
  FLOG(1) << " fd=" << fd << ", offset=" << start << ", size=" << size;
  if (-1 == fd) {
    return -EBADF;
  }

  int result = 0;

  fdpage_list_t new_list;
  if (0 < pagelist->GetUnloadedPages(new_list, start, size)) {
    for (auto iter = new_list.begin(); iter != new_list.end(); ++iter) {
      if ((0 != size) &&
          (static_cast<size_t>(start + size) <= static_cast<size_t>((*iter)->offset))) {
        // Current page is further than the range we were asked to load
        break;
      }

      size_t need_load_size = 0;

      if (static_cast<size_t>((*iter)->offset) < inode->size) {
        need_load_size = (static_cast<size_t>((*iter)->next()) <= inode->size ?
            (*iter)->bytes : (inode->size - (*iter)->offset));
      }
      size_t over_size = (*iter)->bytes - need_load_size;
      if (0 == need_load_size) {
        break;
      }

      if (lseek(fd, (*iter)->offset, SEEK_SET) < 0)
        return -errno;

      FLOG(2) << " fd=" << fd << ", offset=" << start << ", size=" << size
          << " downloading [" << (*iter)->offset << ", "
          << ((*iter)->offset + need_load_size) << "]";
      struct oio_error_s *err;

      struct oio_sds_dl_dst_s dst{};
      dst.type = oio_sds_dl_dst_type_e::OIO_DL_DST_HOOK_SEQUENTIAL;
      dst.data.hook.cb = _write_FILE;
      brun_dl b {fd: fd};
      oio_sds_dl_range_s r {offset: static_cast<size_t>((*iter)->offset),
          size: need_load_size};
      oio_sds_dl_range_s *rtab[] = {&r, nullptr};
      dst.data.hook.ctx = &b;
      dst.data.hook.length = static_cast<size_t>(need_load_size);

      struct oio_sds_dl_src_s src{};
      src.url = url;
      src.ranges = rtab;
      err = oio_sds_download(oio_client, &src, &dst);

      if (err) {
        result = -EIO;
        LOG(ERROR) << "Failed to load data in range [" << r.offset << ", "
            << r.offset + r.size << "]: " << oio_error_message(err);
        oio_error_pfree(&err);
      }

      // TODO handle oio sds error
      if (0 != result) {
        break;
      }

      if (0 < over_size) {
        if (0 != (result = FileHandle::FillFile(fd, 0, over_size,
              (*iter)->offset + need_load_size))) {
          // TODO(noone) error failed to fill fd
          LOG(ERROR) << "failed to fill file fd(" << fd <<"): "
              << strerror(errno);
          break;
        }
        is_modified = false;
      }

      pagelist->SetPageLoadedStatus((*iter)->offset, static_cast<off_t>((*iter)->bytes), true);
    }
    PageList::FreeList(new_list);
  }
  return result;
}

ssize_t FileHandle::Write(const char *buf, off_t start, size_t size) {
  int err = EnsureOpen();
  if (err)
    return err;

  ssize_t result = 0;
  ssize_t wsize;

  FLOG(1) << " fd=" << fd << ", offset=" << start << ", size=" << size;

  size_t restsize = pagelist->GetTotalUnloadedPageSize(0, start) + size;
  if (FileCache::IsSafeDiskSpace(restsize)) {
    if ((0 < start) && 0 != (result = Load(0, static_cast<size_t>(start)))) {
      // TODO(noone) error failed to load before writing
      return result;
    }
  } else {
    return -ENOSPC;
  }
  // Write
  if (-1 == (wsize = pwrite(fd, buf, size, start))) {
    // TODO(noone) pwrite failed. errno
    return -errno;
  }

  is_modified = true;

  if (0 < wsize) {
    pagelist->SetPageLoadedStatus(start, static_cast<size_t>(wsize), true);
  }

  inode->dirty = true;
  GetSize(inode->unflushed_size);

  return wsize;
}

int FileHandle::Truncate(size_t size) {
  int err = EnsureOpen();
  if (err)
    return err;

  size_t current_size;
  GetSize(current_size);

  FLOG(1) << " fd=" << fd << ", size=" << size
      << ", current_size=" << current_size;

  /* We allow truncating the file to something bigger
   * than OIOFS_MAX_FILE_SIZE if for a reason the current size
   * is already bigger and we are asked to shrink the file. */
  if (size > current_size && size > OIOFS_MAX_FILE_SIZE)
    return -EFBIG;

  if (ftruncate(fd, size) < 0) {
    LOG(ERROR) << __FUNCTION__ << " fd=" << fd << ", size=" << size
        << ", errno=" << errno << " " << strerror(errno);
    return -errno;
  }
  pagelist->Resize(size, true);

  is_modified = true;
  inode->dirty = true;
  inode->unflushed_size = size;

  return 0;
}

int FileHandle::Chmod(mode_t new_mode) {
  mode = new_mode;
  int err = EnsureOpen();
  if (err)
    return err;
  return 0;
}

static size_t _read_FILE(void *ctx, unsigned char *ptr, size_t len) {
  FILE *in = static_cast<FILE*>(ctx);
  if (ferror(in)) {
    return OIO_SDS_UL__ERROR;
  }
  if (feof(in)) {
    return OIO_SDS_UL__DONE;
  }
  auto r = fread(ptr, 1, len, in);
  return (0 == r) ? OIO_SDS_UL__NODATA : r;
}

int FileHandle::DataFlush(bool fsync) {
  int rc = EnsureOpen();
  if (rc)
    return rc;

  FLOG(1) << " fd=" << fd << ", is_modified=" << is_modified;

  if (!fsync && !is_modified) {
    return 0;
  }

  int result = 0;

  size_t unloaded_size = pagelist->GetTotalUnloadedPageSize();
  if (0 < unloaded_size) {
    if (FileCache::IsSafeDiskSpace(unloaded_size)) {
      if (0 != (result = Load())) {
        LOG(ERROR) << "failed to upload: " << strerror(errno);
        return static_cast<ssize_t>(result);
      }
    } else {
      // TODO(noone) error failed upload
      LOG(ERROR) << "failed to upload (no disk space available)";
      return -ENOSPC;
    }
  }

  if (0 != lseek(fd, 0, SEEK_SET)) {
    LOG(ERROR) << "lseek error: " << strerror(errno);
    return -errno;
  }

  struct oio_error_s *err;

  struct oio_sds_ul_dst_s dst{};
  dst.url = url;
  dst.autocreate = true;

  int fd2 = -1;
  std::FILE *in = nullptr;
  struct stat st;

  struct oio_sds_ul_src_s src{};
  src.type = oio_sds_ul_src_type_e::OIO_UL_SRC_HOOK_SEQUENTIAL;

  if (-1 == (fd2 = dup(fd))
    || (-1 == fstat(fd2, &st))
    || (0 != lseek(fd2, 0, SEEK_SET))
    || (nullptr == (in = fdopen(fd2, "rb")))) {
    if (-1 != fd2) {
      close(fd2);
    }
    result = -errno;
  } else {
    src.data.hook.cb = _read_FILE;
    final_size = st.st_size;
    src.data.hook.size = final_size;
    src.data.hook.ctx = in;
    err = oio_sds_upload(oio_client, &src, &dst);
    if (err) {
      result = -EIO;
      LOG(ERROR) << "upload error: (" << oio_error_code(err)
          << ") " << oio_error_message(err);
      oio_error_pfree(&err);
    }

    if (in) {
      fclose(in);
    }
    if (-1 != fd2) {
      close(fd2);
    }
  }

  if ((0 == result) && (0 != lseek(fd, 0, SEEK_SET))) {
    LOG(ERROR) << "lseek error: " << strerror(errno);
    return -errno;
  }

  if (0 == result) {
    is_modified = false;
    // Do not clear inode->dirty here, we must update size attr first
  }
  return result;
}

bool FileHandle::GetSize(size_t& size) {
  if (-1 == fd) {
    return false;
  }
  size = pagelist->Size();
  return true;
}

FileCache::FileCache(struct oio_sds_s *oio, struct oio_url_s *url, const std::string &cache_dir) {
  oio_client_ = oio;
  oio_url_ = url;
  cache_dir_ = cache_dir;
  container_name_ = oio_url_get(url, OIOURL_USER);
  std::ostringstream os;
  os << cache_dir_ << "/" << container_name_;
  mkdir(os.str().c_str(), 0755);
}

FileCache::~FileCache() {}

bool FileCache::IsSafeDiskSpace(size_t size) {
  // TODO
  return true;
}

void FileCache::Forget(Inode *in) {
  std::string temp_file;
  MakeCachePath(in, temp_file);
  FLOG(1) << " ino=" << in->ino << ", path=" << temp_file;
  unlink(temp_file.c_str());
}

void FileCache::MakeCachePath(Inode *in, std::string &out) {
  std::ostringstream os;
  os << cache_dir_ << "/" << container_name_ << "/inode-" << in->ino;
  out = os.str();
}

FileHandle* FileCache::CreateFileHandle(Inode *in, int mode) {
  assert(in);

  std::string cache_path{};
  MakeCachePath(in, cache_path);
  FileHandle *fh = new FileHandle(oio_client_, oio_url_dup(in->url),
      cache_path.c_str());
  fh->mode = mode;
  fh->inode = in;
  fh->pagelist = &in->pagelist;
  return fh;
}

bool FileCache::Close(FileHandle *fh) {
  fh->Close();
  if (!fh->IsOpen()) {
    delete fh;
    return true;
  }
  return false;
}
