/*
 *
 * Copyright 2015, OpenIO
 *
 */

#ifndef OIOFS_INTERNAL_FILE_CACHE_H_
#define OIOFS_INTERNAL_FILE_CACHE_H_

#define GLOG_NO_ABBREVIATED_SEVERITIES

#include <glog/logging.h>
#include <sys/stat.h>
#include <assert.h>
#include <unistd.h>
#include <errno.h>
#include <oio_sds.h>
#include <oio_core.h>
#include <list>
#include <map>
#include <memory>
#include <string>
#include "src/inode.h"

struct Inode;

struct FileHandle {
  explicit FileHandle(struct oio_sds_s *oio, struct oio_url_s *oio_url, const char *path);
  ~FileHandle();

  int Open();
  void Close();
  int Load(off_t start = 0, size_t size = 0);
  ssize_t Read(char *buf, off_t start, size_t size, bool force_load = false);
  ssize_t Write(const char *buf, off_t start, size_t size);

  /**
   * If `size` is:
   *  - lower than actual size: shrink the file;
   *  - greater than actual size: grow the file with zeroes;
   *  - equal to actual size: set the 'dirty' flag.
   */
  int Truncate(size_t size);

  int Chmod(mode_t mode);

  int DataFlush(bool fsync = false);
  int Dup();
  bool GetSize(size_t& size);
  bool IsOpen() const { return (-1 != fd); }
  int EnsureOpen();
  void Get() { ++ref; }
  int Put() { return --ref; }

  PageList *pagelist;  // shortcut for inode->pagelist
  Inode *inode;
  int ref;
  int mode;
  int fd;
  std::FILE* pfile;
  int refcnt;
  bool is_modified;
  struct oio_sds_s *oio_client;
  struct oio_url_s *url;
  char *cache_path;
  uint64_t final_size;

  static int FillFile(int fd, unsigned char byte, size_t size, off_t start);
  void Clear();
};

typedef std::map<std::string, class FileHandle*> fh_map_t;

class FileCache {
 public:
  explicit FileCache(struct oio_sds_s *oio, struct oio_url_s *url, const std::string &path);
  ~FileCache();

  void MakeCachePath(Inode *in, std::string &cache_path);
  static bool IsSafeDiskSpace(size_t size);
  FileHandle* CreateFileHandle(Inode *in, int mode);
  bool Close(FileHandle *fh);

  /** Clear from the cache all data related to the inode */
  void Forget(Inode *in);

 private:
  struct oio_sds_s *oio_client_;
  struct oio_url_s *oio_url_;
  std::string cache_dir_;
  std::string container_name_;
};

#endif  // OIOFS_INTERNAL_FILE_CACHE_H_
