/*
 *
 * Copyright 2015, OpenIO
 *
 */

#define FUSE_USE_VERSION 26
#include "src/fuse/oiofs_fuse.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>
#include <dirent.h>
#include <getopt.h>
#include <fuse.h>
#include <fuse/fuse_lowlevel.h>

#include <algorithm>
#include <iostream>
#include <string>
#include <vector>

#include "src/meta/client.h"
#include "src/oiofs_defaults.h"


namespace oiofs {

struct oiofs_fuse_handle {
  oiofs_fuse_handle() {}

  void put_inode(Inode *in) {
    client->PutInode(in);
  }

  Inode* get_inode(fuse_ino_t ino) {
    Inode *in = nullptr;
    client->GetInode(static_cast<ino_t>(ino), &in);
    return in;
  }

  int foreground;
  int verbosity = 0;
  const char *syslog_id = nullptr;
  const char *user_url = nullptr;
  const char *server_addr = nullptr;  // Fetched from meta1
  const char *cache_dir = nullptr;
  int timeout = 30;
  Client *client = nullptr;
};

static void oiofs_ll_statfs(fuse_req_t req, fuse_ino_t ino) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  Inode *in = handle->get_inode(ino);
  struct statvfs stbuf = {0};

  int result = handle->client->Statfs(in, &stbuf);
  if (0 == result) {
    fuse_reply_statfs(req, &stbuf);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(in);
}

static void oiofs_ll_releasedir(fuse_req_t req, fuse_ino_t ino, struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  dir_result_t *dirp = reinterpret_cast<dir_result_t*>(fi->fh);
  handle->client->Releasedir(dirp);
  fuse_reply_err(req, 0);
}

static void oiofs_ll_getattr(fuse_req_t req, fuse_ino_t ino,
    struct fuse_file_info *fi) {
  FileHandle *fh = nullptr;
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *in = handle->get_inode(ino);

  struct stat stbuf = {0};

  /* Documentation says fi is always NULL, but we still get it sometimes */
  if (fi != nullptr && fi->fh != 0)
    fh = reinterpret_cast<FileHandle*>(fi->fh);

  if (fh != nullptr) {
    int err = fh->EnsureOpen();
    if (err) {
      fuse_reply_err(req, -err);  // err is -errno
      return;
    }
  }

  int result = handle->client->Getattr(in, &stbuf, ctx->uid, ctx->gid);
  if (0 == result) {
    if (fh != nullptr) {
      size_t fh_size = 0;
      if (fh->GetSize(fh_size))
        stbuf.st_size = fh_size;
    }
    fuse_reply_attr(req, &stbuf, 0);
  } else {
    fuse_reply_err(req, ENOENT);
  }
  handle->put_inode(in);
}

static void oiofs_ll_setattr(fuse_req_t req, fuse_ino_t ino, struct stat *attr, int to_set,
    struct fuse_file_info *fi) {
  int result = 0;
  FileHandle *fh = nullptr;
  bool fh_opened_locally = false;
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  if (fi != nullptr && fi->fh != 0)
    fh = reinterpret_cast<FileHandle*>(fi->fh);
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *in = handle->get_inode(ino);

  int mask = 0;
  mask |= to_set & (FUSE_SET_ATTR_MODE|FUSE_SET_ATTR_UID|FUSE_SET_ATTR_GID|
      FUSE_SET_ATTR_MTIME|FUSE_SET_ATTR_ATIME|FUSE_SET_ATTR_SIZE|
      FUSE_SET_ATTR_MTIME_NOW|FUSE_SET_ATTR_ATIME_NOW);

  if (mask & (FUSE_SET_ATTR_SIZE|FUSE_SET_ATTR_MODE)) {
    if (fh == nullptr) {
      result = handle->client->Open(in, O_RDWR, 0, ctx->uid, ctx->gid, &fh);
      fh_opened_locally = result == 0;
    }

    if (!result) {
      if (mask & FUSE_SET_ATTR_SIZE)
        result = fh->Truncate(attr->st_size);
      if (!result && mask & FUSE_SET_ATTR_MODE)
        result = fh->Chmod(attr->st_mode);
    }

    if (fh_opened_locally) {
      if (!result)
        result = handle->client->Flush(fh);
      handle->client->Release(fh);
    }
  }

  if (result == 0)
    result = handle->client->Setattr(in, attr, mask, ctx->uid, ctx->gid);

  if (0 == result) {
    fuse_reply_attr(req, attr, 0);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(in);
}

static void oiofs_ll_mkdir(fuse_req_t req, fuse_ino_t parent, const char *name, mode_t mode) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *i1 = handle->get_inode(parent);
  Inode *i2 = nullptr;
  struct fuse_entry_param fe = {0};

  int result;
  result = handle->client->Mkdir(i1, name, mode, &fe.attr, ctx->uid, ctx->gid, &i2);

  if (0 == result) {
    fe.ino = fe.attr.st_ino;
    fuse_reply_entry(req, &fe);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(i1);
}

static void oiofs_ll_lookup(fuse_req_t req, fuse_ino_t parent, const char *name) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  struct fuse_entry_param fe = {0};

  Inode *i1 = handle->get_inode(parent);
  Inode *i2 = nullptr;  // FIXME(noone): i2 has to be freed somewhere
  int result = handle->client->Lookup(i1, name, &fe.attr, ctx->uid, ctx->gid, &i2);
  if (result >= 0) {
    fe.ino = fe.attr.st_ino;
    fe.generation = i2->generation;
    fuse_reply_entry(req, &fe);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(i1);
}

static void oiofs_ll_forget(fuse_req_t req, fuse_ino_t ino, long unsigned nlookup) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  handle->client->ForgetInode(handle->get_inode(ino), nlookup + 1);
  fuse_reply_none(req);
}

static void oiofs_ll_opendir(fuse_req_t req, fuse_ino_t ino, struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *in = handle->get_inode(ino);
  dir_result_t *dirp = nullptr;

  int result = 0;
  result = handle->client->Opendir(in, ctx->uid, ctx->gid, &dirp);

  if (result >= 0) {
    fi->fh = reinterpret_cast<uint64_t>(dirp);
    fuse_reply_open(req, fi);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(in);
}

struct dirbuf {
  char *p;
  size_t size;
};

struct readdir_context {
  fuse_req_t req;
  char *buf;
  size_t size;
  size_t pos;
};

static int oiofs_ll_add_dirent(void *p, struct dirent *de, struct stat *st,
    int mask, off_t next_off) {
  auto c = reinterpret_cast<struct readdir_context*>(p);
  st->st_ino = de->d_ino;

  size_t space_left = c->size - c->pos;
  size_t entry_size = fuse_add_direntry(c->req, c->buf + c->pos, space_left,
      de->d_name, st, next_off);

  if (entry_size > space_left) {
    return -ENOSPC;
  }

  c->pos += entry_size;
  return 0;
}

static void oiofs_ll_readdir(fuse_req_t req, fuse_ino_t ino, size_t size,
    off_t off, struct fuse_file_info *fi) {

  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  dir_result_t *dirp = reinterpret_cast<dir_result_t*>(fi->fh);
  handle->client->Seekdir(dirp, off);

  struct readdir_context rc;
  rc.req = req;
  rc.buf = new char[size];
  rc.size = size;
  rc.pos = 0;

  int err = handle->client->ReaddirRCb(dirp, oiofs_ll_add_dirent, &rc);
  if (0 == err || -ENOSPC == err) {
    fuse_reply_buf(req, rc.buf, rc.pos);
  } else {
    fuse_reply_err(req, -err);
  }
  delete [] rc.buf;
}

static void oiofs_ll_create(fuse_req_t req, fuse_ino_t parent, const char *name, mode_t mode,
    struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  struct fuse_entry_param fe = {0};
  Inode *i1 = handle->get_inode(parent);
  Inode *i2 = nullptr;
  FileHandle *fh = nullptr;

  int result = handle->client->Create(i1, name, mode, fi->flags, &fe.attr, ctx->uid, ctx->gid,
      &i2, &fh);

  if (0 == result) {
    fi->fh = reinterpret_cast<uint64_t>(fh);
    fe.ino = fe.attr.st_ino;
    fuse_reply_create(req, &fe, fi);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(i1);
}

static void oiofs_ll_open(fuse_req_t req, fuse_ino_t ino,
    struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);

  Inode *in = handle->get_inode(ino);
  FileHandle *fh = nullptr;

  int result = handle->client->Open(in, fi->flags, 0, ctx->uid, ctx->gid, &fh);

  if (0 == result) {
    fi->fh = reinterpret_cast<uint64_t>(fh);
    fuse_reply_open(req, fi);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(in);
}

static void oiofs_ll_read(fuse_req_t req, fuse_ino_t ino, size_t size,
    off_t off, struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  FileHandle *fh = reinterpret_cast<FileHandle*>(fi->fh);

  std::vector<char> buf(size);

  int result = handle->client->Read(fh, off, size, buf.data());
  if (result >= 0) {
    fuse_reply_buf(req, buf.data(), buf.size());
  } else {
    fuse_reply_err(req, -result);
  }
}

static void oiofs_ll_write(fuse_req_t req, fuse_ino_t ino, const char *buf, size_t size,
    off_t off, struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  FileHandle *fh = reinterpret_cast<FileHandle*>(fi->fh);
  int result = handle->client->Write(fh, buf, off, size);
  if (0 <= result) {
    fuse_reply_write(req, result);
  } else {
    fuse_reply_err(req, -result);
  }
}

static void oiofs_ll_flush(fuse_req_t req, fuse_ino_t ino, struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  Inode *in = handle->get_inode(ino);
  FileHandle *fh = reinterpret_cast<FileHandle*>(fi->fh);
  int result = handle->client->Flush(fh);
  in->dirty = false;
  fuse_reply_err(req, -result);
}

static void oiofs_ll_release(fuse_req_t req, fuse_ino_t ino, struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  FileHandle *fh = reinterpret_cast<FileHandle*>(fi->fh);
  int result = handle->client->Release(fh);
  fuse_reply_err(req, -result);
}

static void oiofs_ll_fsync(fuse_req_t req, fuse_ino_t ino, int datasync,
    struct fuse_file_info *fi) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  FileHandle *fh = reinterpret_cast<FileHandle*>(fi->fh);
  int result = handle->client->Fsync(fh, datasync);
  fuse_reply_err(req, -result);
}

static void oiofs_ll_readlink(fuse_req_t req, fuse_ino_t ino) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *in = handle->get_inode(ino);
  char buf[PATH_MAX + 1];

  int result;
  result = handle->client->Readlink(in, buf, sizeof(buf), ctx->uid, ctx->gid);

  if (result >= 0) {
    fuse_reply_readlink(req, buf);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(in);
}

static void oiofs_ll_mknod(fuse_req_t req, fuse_ino_t parent, const char *name, mode_t mode,
    dev_t rdev) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  struct fuse_entry_param fe = {0};
  Inode *i1 = handle->get_inode(parent);
  Inode *i2 = nullptr;

  int result;
  result = handle->client->Mknod(i1, name, mode, rdev, &fe.attr, ctx->uid, ctx->gid, &i2);

  if (0 == result) {
    fe.ino = fe.attr.st_ino;
    fuse_reply_entry(req, &fe);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(i1);
}

static void oiofs_ll_unlink(fuse_req_t req, fuse_ino_t parent, const char *name) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *i1 = handle->get_inode(parent);

  int result = handle->client->Unlink(i1, name, ctx->uid, ctx->gid);
  fuse_reply_err(req, -result);
  handle->put_inode(i1);
}

static void oiofs_ll_rmdir(fuse_req_t req, fuse_ino_t parent, const char *name) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *i1 = handle->get_inode(parent);

  int result = handle->client->Rmdir(i1, name, ctx->uid, ctx->gid);
  fuse_reply_err(req, -result);
  handle->put_inode(i1);
}

static void oiofs_ll_symlink(fuse_req_t req, const char *existing, fuse_ino_t parent,
    const char *name) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  struct fuse_entry_param fe = {0};
  Inode *i1 = handle->get_inode(parent);
  Inode *i2 = nullptr;

  int result = handle->client->Symlink(i1, name, existing, &fe.attr, ctx->uid, ctx->gid, &i2);
  if (0 == result) {
    fe.ino = fe.attr.st_ino;
    fuse_reply_entry(req, &fe);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(i1);
}

static void oiofs_ll_rename(fuse_req_t req, fuse_ino_t parent, const char *name,
    fuse_ino_t newparent, const char *newname) {
  std::cout << "newparent " << newparent << " parent " << parent << std::endl;
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *i1 = handle->get_inode(parent);
  Inode *i2 = handle->get_inode(newparent);

  int result = handle->client->Rename(i1, name, i2, newname, ctx->uid, ctx->gid);
  fuse_reply_err(req, -result);

  handle->put_inode(i1);
  handle->put_inode(i2);
}

static void oiofs_ll_link(fuse_req_t req, fuse_ino_t ino, fuse_ino_t newparent,
    const char *newname) {
  struct oiofs_fuse_handle *handle = reinterpret_cast<oiofs_fuse_handle *>(fuse_req_userdata(req));
  const struct fuse_ctx *ctx = fuse_req_ctx(req);
  Inode *i1 = handle->get_inode(ino);
  Inode *i2 = handle->get_inode(newparent);

  struct fuse_entry_param fe = {0};

  int result = handle->client->Link(i1, i2, newname, &fe.attr, ctx->uid, ctx->gid);
  if (0 == result) {
    fe.ino = fe.attr.st_ino;
    fuse_reply_entry(req, &fe);
  } else {
    fuse_reply_err(req, -result);
  }
  handle->put_inode(i1);
  handle->put_inode(i2);
}

static const struct fuse_lowlevel_ops oper = {
  init: 0,
  destroy: 0,
  lookup: oiofs_ll_lookup,
  forget: oiofs_ll_forget,
  getattr: oiofs_ll_getattr,
  setattr: oiofs_ll_setattr,
  readlink: oiofs_ll_readlink,
  mknod: oiofs_ll_mknod,
  mkdir: oiofs_ll_mkdir,
  unlink: oiofs_ll_unlink,
  rmdir: oiofs_ll_rmdir,
  symlink: oiofs_ll_symlink,
  rename: oiofs_ll_rename,
  link: oiofs_ll_link,
  open: oiofs_ll_open,
  read: oiofs_ll_read,
  write: oiofs_ll_write,
  flush: oiofs_ll_flush,
  release: oiofs_ll_release,
  fsync: oiofs_ll_fsync,
  opendir: oiofs_ll_opendir,
  readdir: oiofs_ll_readdir,
  releasedir: oiofs_ll_releasedir,
  fsyncdir: 0,
  statfs: oiofs_ll_statfs,
  setxattr: 0,
  getxattr: 0,
  listxattr: 0,
  removexattr: 0,
  access: 0,
  create: oiofs_ll_create,
  getlk: 0,
  setlk: 0,
  bmap: 0,
  ioctl: 0,
  poll: 0,
  write_buf: 0,
  retrieve_reply: 0,
  forget_multi: 0,
  flock: 0,
  fallocate: 0
};



enum {
  KEY_CACHEDIR,
  KEY_FOREGROUND,
  KEY_HELP,
  KEY_SERVER,
  KEY_SYSLOG,
  KEY_TIMEOUT,
  KEY_USERURL,
  KEY_VERBOSE,
  KEY_VERSION
};

static const struct fuse_opt opts[] {
  FUSE_OPT_KEY("-v",          KEY_VERBOSE),
  FUSE_OPT_KEY("--verbose",   KEY_VERBOSE),
  FUSE_OPT_KEY("-V",          KEY_VERSION),
  FUSE_OPT_KEY("--version",   KEY_VERSION),
  FUSE_OPT_KEY("-h",          KEY_HELP),
  FUSE_OPT_KEY("--help",      KEY_HELP),
  FUSE_OPT_KEY("debug",       KEY_FOREGROUND),
  FUSE_OPT_KEY("-d",          KEY_FOREGROUND),
  FUSE_OPT_KEY("-f",          KEY_FOREGROUND),
  // Match options without '='
  {"--oiofs-cache-dir %s",
    offsetof(struct oiofs_fuse_handle, cache_dir),
    KEY_CACHEDIR},
  {"--oiofs-timeout %d",
    offsetof(struct oiofs_fuse_handle, timeout),
    KEY_TIMEOUT},
  {"--oiofs-user-url %s",
    offsetof(struct oiofs_fuse_handle, user_url),
    KEY_USERURL},
  {"--oiofs-server %s",
    offsetof(struct oiofs_fuse_handle, server_addr),
    KEY_SERVER},
  {"--syslog-id %s",
    offsetof(struct oiofs_fuse_handle, syslog_id),
    KEY_SYSLOG},
  // Match options with '=' (like almost all fuse options)
  {"--oiofs-cache-dir=%s",
    offsetof(struct oiofs_fuse_handle, cache_dir),
    KEY_CACHEDIR},
  {"--oiofs-timeout=%d",
    offsetof(struct oiofs_fuse_handle, timeout),
    KEY_TIMEOUT},
  {"--oiofs-user-url=%s",
    offsetof(struct oiofs_fuse_handle, user_url),
    KEY_USERURL},
  {"--oiofs-server=%s",
    offsetof(struct oiofs_fuse_handle, server_addr),
    KEY_SERVER},
  {"--syslog-id=%s",
    offsetof(struct oiofs_fuse_handle, syslog_id),
    KEY_SYSLOG},
  FUSE_OPT_END
};


static void usage(const char *progname) {
  printf(
      "usage: %s [oio] mountpoint [options]\n"
      "\n"
      "general options:\n"
      "    -o opt,[opt...]        mount options\n"
      "    -h   --help            print help\n"
      "    -V   --version         print version\n"
      "\n"
      "OIOFS options:\n"
      "    -v   --verbose              be verbose\n"
      "    --syslog-id <id>            set syslog id\n"
      "    --oiofs-cache-dir <dir>     set oiofs cache directory\n"
      "    --oiofs-user-url <url>      set oiofs target user URL\n"
      "    --oiofs-server <host:port>  set oiofs server address\n"
      "    --oiofs-timeout <seconds>   set oiofs RPC timeout\n"
      "\n", progname);
}

static struct fuse_session* oiofs_fuse_setup(int argc, char *argv[],
    const struct fuse_lowlevel_ops *ops, size_t ops_size, char **mountpoint,
    int *multithreaded, void *user_data) {
  struct fuse_args args = FUSE_ARGS_INIT(argc, argv);
  struct fuse_chan *ch;
  struct fuse_session *se;

  int foreground;
  int res;

  res = fuse_parse_cmdline(&args, mountpoint, multithreaded, &foreground);

  if (-1 == res) {
    return nullptr;
  }

  ch = fuse_mount(*mountpoint, &args);
  if (!ch) {
    fuse_opt_free_args(&args);
    goto err_free;
  }

  se = fuse_lowlevel_new(&args, ops, ops_size, user_data);
  fuse_opt_free_args(&args);

  if (nullptr == se) {
    goto err_unmount;
  }

  res = fuse_daemonize(foreground);
  if (-1 == res) {
    goto err_unmount;
  }

  res = fuse_set_signal_handlers(se);
  if (-1 == res) {
    goto err_unmount;
  }
  fuse_session_add_chan(se, ch);

  return se;

 err_unmount:
  fuse_unmount(*mountpoint, ch);
  if (se) {
    fuse_session_destroy(se);
  }
 err_free:
  free(*mountpoint);
  return nullptr;
}

static void oiofs_fuse_teardown(struct fuse_session *se, char *mountpoint) {
  struct fuse_chan *ch = fuse_session_next_chan(se, nullptr);
  fuse_remove_signal_handlers(se);
  fuse_unmount(mountpoint, ch);
  fuse_session_destroy(se);
  free(mountpoint);
}

static int oiofs_fuse_main(struct fuse_args *args, void *user_data) {
  struct fuse_session *se;
  char *mountpoint;
  int multithreaded;
  int res;

  se = oiofs_fuse_setup(args->argc, args->argv, &oper,
      sizeof(struct fuse_lowlevel_ops), &mountpoint, &multithreaded, user_data);

  if (nullptr == se) {
    return 1;
  }

  if (multithreaded) {
    res = fuse_session_loop_mt(se);
  } else {
    res = fuse_session_loop(se);
  }
  oiofs_fuse_teardown(se, mountpoint);
  return res ? 1 : 0;
}

static int opt_procs(void *data, const char *arg, int key,
    struct fuse_args *outargs) {
  struct oiofs_fuse_handle *handle = static_cast<oiofs_fuse_handle*>(data);
  switch (key) {
  case FUSE_OPT_KEY_OPT:
    return 1;

  case FUSE_OPT_KEY_NONOPT:
    return 1;

  case KEY_HELP:
    usage(outargs->argv[0]);
    fuse_opt_add_arg(outargs, "-ho");
    oiofs_fuse_main(outargs, data);
    exit(EXIT_FAILURE);

  case KEY_VERBOSE:
    handle->verbosity++;
    return 0;

  case KEY_VERSION:
    fuse_opt_add_arg(outargs, "--version");
    oiofs_fuse_main(outargs, data);
    exit(EXIT_SUCCESS);

  case KEY_FOREGROUND:
    handle->foreground = 1;
    return 1;

  default:
    fprintf(stderr, "internal error\n");
    abort();
  }
}


}  // namespace oiofs

int main(int argc, char *argv[]) {
  google::InitGoogleLogging(argv[0]);
  oio_log_to_syslog();
  oio_log_init_level(GRID_LOGLVL_INFO);


  struct fuse_args args = FUSE_ARGS_INIT(argc, argv);

  oiofs::oiofs_fuse_handle handle;
  if (-1 == fuse_opt_parse(&args, &handle, oiofs::opts, oiofs::opt_procs)) {
    exit(EXIT_FAILURE);
  }

  if (handle.user_url == nullptr)
    handle.user_url = strdup(OIOFS_DEFAULT_USER_URL);
  if (handle.server_addr == nullptr)
    handle.server_addr = g_strdup("");
  if (handle.cache_dir == nullptr)
    handle.cache_dir = strdup(OIOFS_DEFAULT_CACHE_PATH);

  std::string user_url(handle.user_url);
  std::string srv_addr(handle.server_addr);
  std::string cache_dir(handle.cache_dir);
  std::chrono::seconds timeout(handle.timeout);
  oiofs::Client client(user_url, srv_addr, cache_dir, timeout);
  handle.client = &client;

  oio_log_init_level(GRID_LOGLVL_INFO);
  if (handle.syslog_id) {
    oio_log_to_syslog();
    openlog(handle.syslog_id, LOG_NDELAY, LOG_LOCAL0);
  } else {
    oio_log_to_stderr();
  }
  for (int i = 0; i < handle.verbosity; i++) {
    oio_log_verbose();
    // Same as seting GLOG_v environment variable
    fLI::FLAGS_v++;
  }

  if (client.Mount() != 0) {
    exit(EXIT_FAILURE);
  }
  int rc = oiofs::oiofs_fuse_main(&args, &handle);
  printf("Unmounting...\n");
  client.Unmount();
  printf("Unmounted\n");

  free((void*)handle.user_url);
  free((void*)handle.cache_dir);
  free((void*)handle.server_addr);

  return rc;
}
