/*
 *
 * Copyright 2015, OpenIO
 *
 */

#include "src/meta/backend.h"
#include <errno.h>
#include <sys/stat.h>
#include <hiredis/hiredis.h>
#include <chrono>
#include <ctime>
#include <iostream>
#include <sstream>
#include <cstring>
#include <thread>
#include <utility>
#include <vector>
#include "meta.pb.h"

namespace oiofs {

static std::string kDirent_delimiter = " ";
static std::chrono::seconds kRefresh_interval(10);

Backend::~Backend() {
  if (ctx_ != nullptr)
    redisFree(ctx_);
}

bool Backend::refresh_ctx() {
  auto now = std::chrono::steady_clock::now();
  auto diff = std::chrono::duration_cast<std::chrono::seconds>(now - last_refresh_time_);
  if (diff > kRefresh_interval) {
    return true;
  } else {
    return false;
  }
}

int Backend::check_ctx() {
  if ((nullptr != ctx_) && !refresh_ctx()) {
    return 0;
  }

  redisReply *reply;
  if (nullptr != ctx_) {
    reply = static_cast<redisReply*>(redisCommand(ctx_, "PING"));
    if ((nullptr != reply) && (nullptr != reply->str) && (0 == std::strcmp(reply->str, "PONG"))) {
      freeReplyObject(reply);
      goto refresh;
    } else {
      if (nullptr != reply) {
        freeReplyObject(reply);
      }
    }
  }

  if (!sentinels_.empty()) {
    LOG(INFO) << "Getting master Redis address with Sentinel";
    int retries = sentinels_.size() * 2;
    std::pair<std::string, int> addr;
    do {
      addr = sentinels_.front();
      LOG(INFO) << "Connecting to Sentinel at " << addr.first
          << ":" << addr.second << " (" << retries << " remaining)";
      redisContext *senti_ctx = redisConnect(addr.first.c_str(), addr.second);
      if (senti_ctx && !senti_ctx->err) {
        LOG(INFO) << "Asking for master of cluster '" << sentinel_master_name_
            << "'";
        std::string cmd = "SENTINEL get-master-addr-by-name " +
            sentinel_master_name_;
        reply = static_cast<redisReply*>(redisCommand(senti_ctx, cmd.c_str()));
        if (reply != nullptr &&
            reply->type == REDIS_REPLY_ARRAY &&
            reply->elements >= 2) {
          if (reply->element[0]->type == REDIS_REPLY_STRING) {
            redis_addr_ = reply->element[0]->str;
          }
          if (reply->element[1]->type == REDIS_REPLY_STRING) {
            redis_port_ = std::atoi(reply->element[1]->str);
          } else if (reply->element[1]->type == REDIS_REPLY_INTEGER) {
            redis_port_ = reply->element[1]->integer;
          }
        }
        freeReplyObject(reply);
        LOG(INFO) << "Connecting to " << redis_addr_ << ":" << redis_port_;
        ctx_ = redisConnect(redis_addr_.c_str(), redis_port_);
        if (!ctx_ || ctx_->err) {
          LOG(ERROR) << "Failed: " << (ctx_? ctx_->errstr : strerror(errno))
              << ". Sentinel gave us a bad address! Sleep 1s.";
          redisFree(ctx_);
          ctx_ = nullptr;
          std::this_thread::sleep_for(std::chrono::seconds(1));
        }
      } else if (senti_ctx) {
        LOG(ERROR) << "Failed: (" << senti_ctx->err
            << ") " << senti_ctx->errstr;
      } else {
        LOG(ERROR) << "Failed";
      }
      redisFree(senti_ctx);

      // Round-robin the sentinel addresses
      sentinels_.pop_front();
      sentinels_.push_back(addr);
    } while (--retries > 0 && (!ctx_ || ctx_->err));
  } else {
    ctx_ = redisConnect(redis_addr_.c_str(), redis_port_);
  }
  if (ctx_ == nullptr || ctx_->err) {
    redisFree(ctx_);
    ctx_ = nullptr;
    return -1;
  }
 refresh:
  last_refresh_time_ = std::chrono::steady_clock::now();
  return 0;
}

int Backend::GetInodeStat(const fsid_t &fsid, ino_t ino, InodeStat **out) {
  if (0 != check_ctx()) {
    return -EAGAIN;
  }

  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    "local inode_stat = redis.call('HMGET', fsid..':inode:'..ino,"
    " 'mode', 'uid', 'gid', 'size', 'atime', 'ctime', 'mtime',"
    " 'nlink', 'generation', 'symlink');"
    "return inode_stat";
  auto reply = static_cast<redisReply*>(redisCommand(ctx_, "EVAL %s 2 %s %d", cmd, fsid.c_str(), ino));

  int result = 0;
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_ARRAY) && (reply->elements == 10)) {
    std::array<int, 9> v;
    for (size_t i = 0; i < reply->elements - 1; i++) {
      auto p = reply->element[i];
      if ((nullptr != p) && p->type == REDIS_REPLY_STRING) {
        v[i] = std::atoi(p->str);
      } else {
        result = -ENOENT;
        goto err;
      }
    }
    auto in = new InodeStat;
    in->set_ino(ino);
    in->set_mode(v.at(0));
    in->set_uid(v.at(1));
    in->set_gid(v.at(2));
    in->set_size(v.at(3));
    in->set_atime(v.at(4));
    in->set_ctime(v.at(5));
    in->set_mtime(v.at(6));
    in->set_nlink(v.at(7));
    in->set_generation(v.at(8));
    auto symlink = reply->element[9];
    if ((nullptr != symlink) && symlink->type == REDIS_REPLY_STRING) {
      in->set_symlink(symlink->str, symlink->len);
    }
    *out = in;
  } else {
    result = -EIO;
  }
 err:
  freeReplyObject(reply);
  return result;
}

int Backend::SetInodeStat(const fsid_t &fsid, ino_t ino, InodeStat *inode) {
  if (0 != check_ctx()) {
    return -EAGAIN;
  }

  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    "local mode = KEYS[3];"
    "local size = KEYS[4];"
    "local uid = KEYS[5];"
    "local gid = KEYS[6];"
    "local mtime = KEYS[7];"
    "local ctime = KEYS[8];"
    "local atime = KEYS[9];"
    "local exist = redis.call('EXISTS', fsid..':inode:'..ino);"
    "if not exist then return false end;"
    "redis.call('HMSET', fsid..':inode:'..ino, 'mode', mode,"
    " 'uid', uid, 'gid', gid,"
    " 'mtime', mtime, 'ctime', ctime, 'atime', atime,"
    " 'size', size);"
    "return 1";
  auto reply = static_cast<redisReply*>( redisCommand(ctx_,
        "EVAL %s 9 %s %d %d %d %d %d %d %d %d", cmd, fsid.c_str(), ino,
        inode->mode(), inode->size(), inode->uid(), inode->gid(),
        inode->mtime(), inode->ctime(), inode->atime()));

  int result = 0;
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    if (0 == reply->integer) {
      result = -EEXIST;
      goto err;
    }
  } else {
    result = -EIO;
    goto err;
  }
 err:
  freeReplyObject(reply);
  return result;
}

int Backend::LookupInodeStat(const fsid_t &fsid, ino_t dir, const std::string &path, InodeStat **out) {
  if (0 != check_ctx()) {
    return -EIO;
  }

  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    "local name = KEYS[3];"
    "local lookup_ino = redis.call('HGET', fsid..':dirent:'..ino..':index', name);"
    "if not lookup_ino then return false end;"
    "local info = redis.call('HMGET', fsid..':inode:'..lookup_ino, 'mode', 'uid', 'gid',"
    " 'size', 'atime', 'ctime', 'mtime', 'nlink', 'generation');"
    "info[10] = lookup_ino;"
    "info[11] = redis.call('HLEN', fsid..':dirent:'..lookup_ino..':index');"
    "info[12] = redis.call('HGET', fsid..':inode:'..lookup_ino, 'symlink');"
    "return info";
  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 3 %s %d %s", cmd, fsid.c_str(), dir, path.c_str()));

  int err = 0;
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_ARRAY) && (reply->elements == 12)) {
    std::array<int, 10> v;
    for (size_t i = 0; i < reply->elements - 2; i++) {
      auto p = reply->element[i];
      if ((nullptr != p) && p->type == REDIS_REPLY_STRING) {
        v[i] = std::atoi(p->str);
      } else {
        err = -EIO;
        goto out;
      }
    }
    auto in = new InodeStat;
    in->set_mode(v.at(0));
    in->set_uid(v.at(1));
    in->set_gid(v.at(2));
    in->set_size(v.at(3));
    in->set_atime(v.at(4));
    in->set_ctime(v.at(5));
    in->set_mtime(v.at(6));
    in->set_nlink(v.at(7));
    in->set_generation(v.at(8));
    in->set_ino(v.at(9));
    auto entries = reply->element[10];
    if ((nullptr != entries) && entries->type == REDIS_REPLY_INTEGER) {
      in->set_entries(entries->integer);
    }
    auto symlink = reply->element[11];
    if ((nullptr != symlink) && symlink->type == REDIS_REPLY_STRING) {
      in->set_symlink(symlink->str, symlink->len);
    }
    *out = in;
  } else {
    err = -ENOENT;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::DeallocateInode(const fsid_t &fsid, ino_t ino) {
  if (0 != check_ctx()) {
    return -EIO;
  }

  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    // check nlink
    "local nlink = redis.call('HGET', fsid..':inode:'..ino, 'nlink');"
    "if tonumber(nlink) ~= 0 then return tonumber(nlink) end;"
    // delete inode
    "redis.call('DEL', fsid..':inode:'..ino);"
    // deallocate inode in bitmap
    "redis.call('SETBIT', fsid..':inode_bitmap', ino, 0);"
    "return 0;";
  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 2 %s %d", cmd, fsid.c_str(), ino));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    // TODO wat to?
    err = reply->integer;
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}


int Backend::AllocateInode(const fsid_t &fsid, mode_t mode, uid_t uid, gid_t gid, InodeStat **out) {
  if (0 != check_ctx()) {
    return -EIO;
  }

  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local mode = KEYS[2];"
    "local uid = KEYS[3];"
    "local gid = KEYS[4];"
    "local time = KEYS[5];"
    // find unused ino from bitmap
    "local ino = redis.call('BITPOS', fsid..':inode_bitmap', 0);"
    "if ino == -1 then return false end;"
    // allocate ino in bitmap
    "redis.call('SETBIT', fsid..':inode_bitmap', ino, 1);"
    // create inode
    "redis.call('HMSET', fsid..':inode:'..ino, 'mode', mode, 'uid', uid, 'gid', gid,"
    " 'size', 0, 'atime', time, 'ctime', time, 'mtime', time, 'nlink', 0,"
    " 'generation', time);"
    // return allocated ino
    "return ino";
  auto ctime = std::time(nullptr);
  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 5 %s %d %d %d %d", cmd, fsid.c_str(), mode, uid, gid, ctime));

  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    InodeStat *new_inode = new InodeStat();
    new_inode->set_mode(mode);
    new_inode->set_uid(uid);
    new_inode->set_gid(gid);
    new_inode->set_size(0);
    new_inode->set_atime(ctime);
    new_inode->set_ctime(ctime);
    new_inode->set_mtime(ctime);
    new_inode->set_nlink(1);
    new_inode->set_generation(ctime);
    new_inode->set_ino(reply->integer);
    *out = new_inode;
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::SetSymlink(const fsid_t &fsid, ino_t ino, const std::string &target) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    "local target = KEYS[3];"
    "redis.call('HSET', fsid..':inode:'..ino, 'symlink', target);"
    "return 1;";
  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 3 %s %d %s", cmd, fsid.c_str(), ino, target.c_str()));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    // TODO wtf?
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::DelDir(const fsid_t &fsid, ino_t ino) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    // verify empty dir
    "local n = redis.call('HLEN', fsid..':dirent:'..ino..':index');"
    // . and .. dentries minimum entries
    "if n ~= 2 then return 0 end;"
    // delete dir index
    "redis.call('DEL', fsid..':dirent:'..ino..':index');"
    // delete dir list
    "redis.call('DEL', fsid..':dirent:'..ino);"
    "return 1;";
  auto reply = static_cast<redisReply*>(redisCommand(ctx_, "EVAL %s 2 %s %d", cmd, fsid.c_str(), ino));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    if (0 == reply->integer) {
      err = -ENOTEMPTY;
      goto out;
    }
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::AddDir(const fsid_t &fsid, ino_t parent, const std::string &name, ino_t ino) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local parent = KEYS[2];"
    "local name = KEYS[3];"
    "local ino = KEYS[4];"
    "redis.call('RPUSH', fsid..':dirent:'..ino, ino..' .');"
    "redis.call('RPUSH', fsid..':dirent:'..ino, parent..' ..');"
    "redis.call('HSET', fsid..':dirent:'..ino..':index', '.' , ino);"
    "redis.call('HSET', fsid..':dirent:'..ino..':index', '..' , parent);"
    "return 1;";
  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 4 %s %d %s %d", cmd, fsid.c_str(), parent, name.c_str(), ino));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    if (0 == reply->integer) {
      err = -EEXIST;
      goto out;
    }
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::AddLink(const fsid_t &fsid, ino_t parent, const std::string &name, ino_t ino) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local parent = KEYS[2];"
    "local name = KEYS[3];"
    "local ino = KEYS[4];"
    "local exist = redis.call('HSETNX', fsid..':dirent:'..parent..':index', name, ino);"
    "if exist == 0 then return exist end;"
    "local dirent = ino..' '..name;"
    "return redis.call('RPUSH', fsid..':dirent:'..parent, dirent)";

  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 4 %s %d %s %d", cmd, fsid.c_str(), parent, name.c_str(), ino));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    if (0 == reply->integer) {
      err = -EEXIST;
      goto out;
    }
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::parse_dirent(const char *str, struct dirent *d) {
  size_t pos = 0;
  std::string s(str);
  std::string token;
  pos = s.find(kDirent_delimiter);
  token = s.substr(0, pos);
  s = s.substr(pos + kDirent_delimiter.length());
  std::strncpy(d->d_name, s.c_str(), 255);
  d->d_name[255] = '\0';
  d->d_ino = ::atoi(token.c_str());
  return 0;
}

int Backend::SetLink(const fsid_t &fsid, ino_t parent, const std::string &name, const std::string &new_name,
    ino_t new_ino) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local parent = KEYS[2];"
    "local name = KEYS[3];"
    "local new_name = KEYS[4];"
    "local new_ino = KEYS[5];"
    // lookup ino
    "local ino = redis.call('HGET', fsid..':dirent:'..parent..':index', name);"
    "local old_ino = redis.call('HGET', fsid..':dirent:'..parent..':index', new_name);"
    // remove entry from index
    "redis.call('HDEL', fsid..':dirent:'..parent..':index', name);"
    "redis.call('HSET', fsid..':dirent:'..parent..':index', new_name, new_ino);"
    // remove entry from dir
    "redis.call('LREM', fsid..':dirent:'..parent, 1, old_ino..' '..new_name);"
    "local dirent = new_ino..' '..new_name;"
    "redis.call('RPUSH', fsid..':dirent:'..parent, dirent)"
    "return 1";

  auto reply = static_cast<redisReply*>( redisCommand(ctx_, "EVAL %s 5 %s %d %s %s %d",
      cmd, fsid.c_str(), parent, name.c_str(), new_name.c_str(), new_ino));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    if (0 == reply->integer) {
      err = -EEXIST;
      goto out;
    }
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::DelLink(const fsid_t &fsid, ino_t parent, const std::string &name, ino_t ino) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local parent = KEYS[2];"
    "local name = KEYS[3];"
    "local ino = KEYS[4];"
    // lookup lookup_ino
    "local lookup_ino = redis.call('HGET', fsid..':dirent:'..parent..':index', name);"
    // remove entry from dir
    "local dirent = ino..' '..name;"
    "redis.call('LREM', fsid..':dirent:'..parent, 1, dirent);"
    // remove entry from index
    "if lookup_ino ~= ino then return 1 end;"
    "redis.call('HDEL', fsid..':dirent:'..parent..':index', name);"
    "return 1";

  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 4 %s %d %s %d", cmd, fsid.c_str(), parent, name.c_str(), ino));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
    if (0 == reply->integer) {
      err = -EEXIST;
      goto out;
    }
  } else {
    err = -EIO;
    goto out;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::Readdir(const fsid_t &fsid, ino_t ino, loff_t off, dirent_cb_t cb) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    "local offset = KEYS[3];"
    "local dirents = redis.call('LRANGE', fsid..':dirent:'..ino, offset, 1000);"
    "return dirents;";

  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 3 %s %d %d", cmd, fsid.c_str(), ino, off));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_ARRAY)) {
    for (size_t i = 0; i < reply->elements; i++) {
      auto p = reply->element[i];
      if ((nullptr != p) && (p->type == REDIS_REPLY_STRING)) {
        struct dirent d{};
        err = parse_dirent(p->str, &d);
        if (err) {
          goto out;
        }
        cb(d.d_name, d.d_ino);
      } else {
        err = -ENOENT;
        goto out;
      }
    }
  } else {
    err = -EIO;
  }
 out:
  freeReplyObject(reply);
  return err;
}

int Backend::IncrNlink(const fsid_t &fsid, ino_t ino, int count) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  const char *cmd =
    "local fsid = KEYS[1];"
    "local ino = KEYS[2];"
    "local count = KEYS[3];"
    "return redis.call('HINCRBY', fsid..':inode:'..ino, 'nlink', count);";

  auto reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 3 %s %d %d", cmd, fsid.c_str(), ino, count));
  if ((nullptr != reply) && (reply->type == REDIS_REPLY_INTEGER)) {
  } else {
    err = -EIO;
  }
  freeReplyObject(reply);
  return err;
}

int Backend::Mkfs(const fsid_t &fsid) {
  if (0 != check_ctx()) {
    return -EIO;
  }
  int err = 0;
  // Clean inode_bitmap and other fsid-related entries and reserve inode 0
  const char *cmd =
    "local fsid = KEYS[1];"
    "redis.call('DEL', fsid..':inode_bitmap', unpack(redis.call('KEYS', fsid..':*')));"
    "redis.call('SETBIT', fsid..':inode_bitmap', 0, 1);"
    "return 1";
  redisReply *reply = static_cast<redisReply*>(
      redisCommand(ctx_, "EVAL %s 1 %s", cmd, fsid.c_str()));
  if (reply == nullptr ||
      reply->type != REDIS_REPLY_INTEGER ||
      reply->integer != 1) {
    err = -EIO;
  }
  freeReplyObject(reply);
  // Create "/" as inode 1
  if (!err) {
    InodeStat *new_inode = nullptr;
    DelDir(fsid, 1);
    err = AllocateInode(fsid, S_IFDIR | 0755, 0, 0, &new_inode);
    if (!err) {
      err = AddDir(fsid, 1, "", new_inode->ino());  // name is useless
      if (!err) {
        err = SetInodeStat(fsid, new_inode->ino(), new_inode);
      }
      delete new_inode;
    }
  }
  return err;
}

}  // namespace oiofs
