/*
 *
 * Copyright 2015, OpenIO
 *
 */

#ifndef OIOFS_INTERNAL_META_BACKEND_H_
#define OIOFS_INTERNAL_META_BACKEND_H_

#include <hiredis/hiredis.h>
#include <dirent.h>
#include <chrono>
#include <functional>
#include <list>
#include <string>
#include <utility>
#include "./meta.pb.h"

#define GLOG_NO_ABBREVIATED_SEVERITIES
#include <glog/logging.h>

/*
 * Inode
 *
 * -- Fields --
 * mode: file type and access rights
 * uid: owner id
 * gid: group id
 * size: file size in bytes
 * atime: last access time
 * ctime: inode last modification time
 * mtime: file last modification time
 * nlink: link count
 * symlink: symlink content
 * generation: file version
 * url: object url
 *
 * -- Internal --
 * key name: $fsid:inode:$ino
 * key type: redis hash
 * hash keys: inode fields
 * hash values: fields values
 */

/*
 *
 * Directory
 *
 * List
 *
 * -- Internal --
 * key name: $fsid:dirent:$ino
 * key type: redis list
 * list values: dirent
 *
 * dirent format: "$ino $name"
 *
 * Note:
 * An empty directory always contains
 * . and .. entries.
 *
 *
 * Index (fast lookup)
 *
 * -- Internal --
 * key name: $fsid:dirent:$ino:index
 * key type: redis hash
 * hash keys: dirent name
 * hash values: dirent ino
 *
 */


namespace oiofs {

typedef std::string fsid_t;
typedef std::function<int(const char*, ino_t)> dirent_cb_t;

class Backend {
 public:
  explicit Backend(std::string redis_addr, int redis_port,
      std::list<std::pair<std::string, int>> sentinels,
      std::string sentinel_master_name)
    : redis_addr_(redis_addr), redis_port_(redis_port),
      sentinel_master_name_(sentinel_master_name), sentinels_(sentinels),
      ctx_(nullptr) {}
  ~Backend();

  /**
   * Get inode info.
   *
   * @param fsid the filesystem fsid_t
   * @param ino the inode's ino_t
   * @param out InodeStat result out if success
   * @returns 0 on success, negative error code on failure
   */
  int GetInodeStat(const fsid_t &fsid, ino_t ino, InodeStat **out);

  /**
   * Set inode info.
   *
   * @param fsid the filesystem fsid_t
   * @param ino the inode's ino_t
   * @param inode the stat to set
   * @returns 0 on success, negative error code on failure
   */
  int SetInodeStat(const fsid_t &fsid, ino_t ino, InodeStat *inode);

  /**
   * Lookup inode info.
   *
   * @param fsid the filesystem fsid_t
   * @param ino the directory's ino_t
   * @param name name to lookup in the directory
   * @param out InodeStat result out if success
   * @returns 0 on success, negative error code on failure
   */
  int LookupInodeStat(const fsid_t &fsid, ino_t dir, const std::string &name, InodeStat **out);

  int DeallocateInode(const fsid_t &fsid, ino_t ino);

  int AllocateInode(const fsid_t &fsid, mode_t mode, uid_t uid, gid_t gid, InodeStat **out);

  int SetSymlink(const fsid_t &fsid, ino_t ino, const std::string &target);

  int AddDir(const fsid_t &fsid, ino_t parent, const std::string &name, ino_t ino);

  int AddLink(const fsid_t &fsid, ino_t parent, const std::string &name, ino_t ino);

  int SetLink(const fsid_t &fsid, ino_t parent, const std::string &name, const std::string &new_name, ino_t new_ino);

  int DelLink(const fsid_t &fsid, ino_t parent, const std::string &name, ino_t ino);

  int Readdir(const fsid_t &fsid, ino_t ino, loff_t offset, dirent_cb_t cb);

  int DelDir(const fsid_t &fsid, ino_t ino);

  int IncrNlink(const fsid_t &fsid, ino_t ino, int count);

  /**
   * Initialize inode 0 and 1.
   *
   * @param fsid the filesystem fsid_t (usually a container URL)
   */
  int Mkfs(const fsid_t &fsid);

 private:
  bool refresh_ctx();
  int check_ctx();
  int parse_dirent(const char *str, struct dirent *d);

  std::string redis_addr_;
  int redis_port_;
  std::string sentinel_master_name_;
  std::list<std::pair<std::string, int>> sentinels_;

  redisContext *ctx_;
  std::chrono::steady_clock::time_point last_refresh_time_;
};
}  // namespace oiofs

#endif  // OIOFS_INTERNAL_META_BACKEND_H_
