/*
 *
 * Copyright 2015, OpenIO
 *
 */

#include "src/meta/client.h"

#include <glib.h>
#include <oio_core.h>

#include <sys/statfs.h>

#include <grpc++/grpc++.h>
#include <fcntl.h>
#include <iostream>
#include <string>
#include "./meta.grpc.pb.h"

#ifndef FLOG
# define FLOG(level) VLOG(level) << __FUNCTION__
#endif


namespace oiofs {

dir_result_t::dir_result_t(Inode *in)
  : inode(in), offset(0), next_offset(2), release_count(0), end(false),
    buffer(nullptr) {}

/**
 * Get 3rd field of service descriptions
 * (e.g. "127.0.0.1:4341" from "1,oiofs,127.0.0.1:4341").
 */
static void parse_service_addr(const std::string &in, std::string *out) {
  size_t start = 0, end = 0;
  start = in.find(',', start);
  start = in.find(',', start+1);
  end = in.find(',', start+1);
  *out = in.substr(start+1, end != std::string::npos? end-start : end);
}


Client::Client(
    std::string user_url,
    std::string srv_addr,
    std::string cache_dir,
    std::chrono::seconds timeout)
  : connected_(false),
    mounted_(false),
    user_url_(user_url),
    oio_url_(nullptr),
    server_addr_(srv_addr),
    cache_dir_(cache_dir),
    cache_(nullptr),
    oio_client_(nullptr),
    timeout_(timeout) {
}

Client::~Client() {
  oio_sds_pfree(&oio_client_);
  oio_url_pclean(&oio_url_);
}

void Client::_PrepareContext(grpc::ClientContext *context) {
  auto now = std::chrono::system_clock::now() + timeout_;
  context->set_deadline(now);
}

Inode* Client::handle_inodestat(const InodeStat *st) {
  Inode *in = nullptr;
  bool was_new = false;
  if (inode_map_.count(st->ino())) {
    in = inode_map_[st->ino()];
    // TODO(noone) log debug
  } else {
    in = new Inode(st->ino(), url_from_ino(st->ino()));
    inode_map_[st->ino()] = in;

    // if (!root_inode_) {
    //   root_inode_ = in;
    // }

    in->ino = st->ino();
    in->mode = st->mode() & S_IFMT;
    was_new = true;
  }

  in->generation = st->generation();
  if (in->is_symlink()) {
    in->symlink = st->symlink();
  }

  // update inode stat
  in->mode = st->mode();
  in->uid = st->uid();
  in->gid = st->gid();
  in->nlink = st->nlink();
  in->size = st->size();
  in->atime = st->atime();
  in->mtime = st->mtime();
  in->ctime = st->ctime();

  if (was_new) {
    // TODO(noone) log debug added inode
  }
  return in;
}

int Client::Init() {
  return 0;
}

int Client::_Connect() {
  if (oio_url_ == nullptr) {
    oio_url_ = oio_url_init(user_url_.c_str());
    if (oio_url_ == nullptr)
      return -EINVAL;
  }

  if (oio_client_ == nullptr) {
    struct oio_error_s *err;
    err = oio_sds_init(&oio_client_, oio_url_get(oio_url_, OIOURL_NS));
    if (err != nullptr) {
      LOG(ERROR) << "Failed to create SDS client: (" << oio_error_code(err)
          << ") " << oio_error_message(err);
      oio_error_pfree(&err);
      return -EIO;
    }
  }

  if (server_addr_.empty()) {
    GError *gerr = nullptr;
    gchar **services = nullptr;
    struct oio_directory_s *dir = nullptr;
    dir = oio_directory__create_proxy(oio_url_get(oio_url_, OIOURL_NS));
    gerr = oio_directory__list(dir, oio_url_, OIOFS_SRV_TYPE,
        nullptr, &services);
    oio_directory__destroy(dir);
    if (gerr != nullptr) {
      LOG(ERROR) <<  "Failed to get oiofs service for " << user_url_ << ": ("
          << gerr->code << ") " << gerr->message;
      g_clear_error(&gerr);
      return -EIO;
    } else if (services == nullptr || services[0] == nullptr) {
      LOG(ERROR) << "No oiofs service linked to " << user_url_;
      if (services != nullptr)
        g_strfreev(services);
      return -EIO;
    } else {
      // There should be only one
      std::string srv_entry = std::string(services[0]);
      parse_service_addr(srv_entry, &server_addr_);
      g_strfreev(services);
    }
  }

  channel_ = grpc::CreateChannel(server_addr_, grpc::InsecureChannelCredentials());
  stub_ = MetaService::NewStub(channel_);

  connected_ = true;
  return 0;
}

int Client::Mount() {
  std::lock_guard<std::mutex> lock(client_lock_);

  if (mounted_) {
    // TODO(noone) log already mounted
    return 0;
  }

  int rc = 0;
  if (!connected_ && (rc = _Connect()) != 0) {
    return rc;
  }

  if (cache_ == nullptr) {
    const char *cache_dir_str = cache_dir_.c_str();
    struct stat cache_dir_stat;
    if (stat(cache_dir_str, &cache_dir_stat) < 0) {
      if (errno == ENOENT) {
        if (mkdir(cache_dir_str, 0755) < 0) {
          LOG(ERROR) << "Failed to create cache directory " << cache_dir_str
              << ": " << strerror(errno);
          return -EIO;
        }
      } else {
        LOG(ERROR) << "Failed to stat cache directory " << cache_dir_str
            << ": " << strerror(errno);
        return -EIO;
      }
    } else if (!S_ISDIR(cache_dir_stat.st_mode)) {
      LOG(ERROR) << cache_dir_str << " is not a directory";
      return -EIO;
    } else if (access(cache_dir_str, W_OK)) {
      LOG(ERROR) <<  "Cannot write in " << cache_dir_str << ": "
          << strerror(errno);
      return -EIO;
    }
    std::unique_ptr<FileCache> file_cache(
        new FileCache(oio_client_, oio_url_, cache_dir_));
    cache_ = std::move(file_cache);
  }

  // getattr request for root
  GetattrRequest request;
  request.set_fsid(user_url_);
  request.set_ino(OIOFS_INO_ROOT);

  GetattrReply reply;
  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  // TODO(noone) fix that shit
  auto status = stub_->Getattr(&ctx, request, &reply);

  if (!status.ok()) {
    // TODO(noone) error management
    return -EIO;
  }

  // TODO(noone) deal with response

  root_inode_ = new Inode(OIOFS_INO_ROOT, url_from_ino(OIOFS_INO_ROOT));
  root_inode_->mode = S_IFDIR;
  inode_map_[OIOFS_INO_ROOT] = root_inode_;
  mounted_ = true;
  return 0;
}

void Client::Unmount() {
  std::lock_guard<std::mutex> lock(client_lock_);

  for (auto iter = inode_map_.begin();
      iter != inode_map_.end();
      // put_inode will erase the entry, do not use iter++
      iter = inode_map_.begin()) {
    put_inode((*iter).second, 1);
  }

  mounted_ = false;
  delete root_inode_;
}

void Client::Shutdown() {
  // TODO(noone) shutdown client
}

int Client::GetInode(ino_t ino, Inode** out) {
  std::lock_guard<std::mutex> lock(client_lock_);

  auto it = inode_map_.find(ino);
  if (it == inode_map_.end()) {
    *out = nullptr;
    return -1;
  }
  Inode *in = it->second;
  ll_get_inode(in);
  *out = in;
  return 0;
}

void Client::ll_get_inode(Inode *in) {
  in->ref();
}

int Client::ll_put_inode(Inode *in, int count) {
  int refcount = in->unref(count);
  if (0 == refcount) {
    put_inode(in, 1);
  }
  return refcount;
}

int Client::_PerformForget(Inode *in) {
  if (in->ino == OIOFS_INO_ROOT)
    return 0;

  ForgetRequest request;
  request.set_fsid(user_url_);
  request.set_ino(in->ino);

  ForgetReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Forget(&ctx, request, &reply);

  int result;
  if (status.ok()) {
    result = reply.result();
    // TODO handle delete ???
  } else {
    result = -EIO;
  }
  return result;
}

void Client::put_inode(Inode *in, int count) {
  int nlinks = _PerformForget(in);

  if (nlinks == 0 && S_ISREG(in->mode)) {
    FLOG(1) << " inode " << in->ino << " nlinks dropped to zero, deleting";
    struct oio_error_s *err = oio_sds_delete(oio_client_, in->url);
    if (err != NULL && oio_error_code(err) != 404) {
      LOG(ERROR) << "Failed to delete inode " << in->ino
          << ": (" << oio_error_code(err) << ") " << oio_error_message(err);
    }
    oio_error_pfree(&err);
  }
  inode_map_.erase(in->ino);
  if (in == root_inode_) {
    root_inode_ = nullptr;
  }
  cache_->Forget(in);
  delete in;
}

bool Client::PutInode(Inode *in) {
  return ForgetInode(in, 1);
}

bool Client::ForgetInode(Inode *in, int count) {
  std::lock_guard<std::mutex> lock(client_lock_);
  FLOG(1) << " ino=" << in->ino << ", count=" << count;

  assert(nullptr != in);

  ino_t ino = in->ino;

  if (ino == OIOFS_INO_ROOT) {
    // root inode
    return true;
  }

  bool last = false;
  if (in->refcount < count) {
    LOG(ERROR) << "ForgetInode on ino " << in->ino << " count " << count
              << " with refcount=" << in->refcount;
    ll_put_inode(in, in->refcount);
    last = true;
  } else if (0 == ll_put_inode(in, count)) {
    last = true;
  } else {
    // noop
  }
  return last;
}

int Client::_PerformLookup(Inode *dir, const std::string &name,
    int uid, int gid, Inode **out) {
  LookupRequest request;
  request.set_fsid(user_url_);
  request.set_name(name);
  request.set_ino(dir->ino);

  LookupReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Lookup(&ctx, request, &reply);

  int result;
  if (status.ok()) {
    result = reply.result();
    if (0 == result) {
      *out = handle_inodestat(&reply.inode());
    }
  } else {
    result = -EIO;
  }
  return result;
}

int Client::_Lookup(Inode *dir, const std::string &name, int uid, int gid, Inode **out) {
  int result;

  if (!dir->is_dir()) {
    result = -ENOTDIR;
    goto done;
  }

  if (name == "..") {
    // TODO (noone) deal with ..
    result = -ENOENT;
    goto done;
  }

  if (name == ".") {
    *out = dir;
    goto done;
  }

  if (name.length() > NAME_MAX) {
    result = -ENAMETOOLONG;
    goto done;
  }

  result = _PerformLookup(dir, name, uid, gid, out);

 done:
  // TODO(noone) debug log
  return result;
}

int Client::Lookup(Inode *parent, const std::string &name, struct stat *attr,
    int uid, int gid, Inode **out) {
  std::lock_guard<std::mutex> lock(client_lock_);
  FLOG(1) << " name=" << name << ", uid=" << uid << ", gid=" << gid;

  assert(nullptr != parent);

  Inode *inp = nullptr;
  int result;
  result = _Lookup(parent, name, uid, gid, &inp);
  if (result < 0) {
    attr->st_ino = 0;
    goto end;
  }

  assert(inp);
  fill_stat(inp, attr);
  ll_get_inode(inp);

  *out = inp;
 end:
  return result;
}

int Client::_Open(Inode *in, int flags, mode_t mode, int uid, int gid, FileHandle **out) {
  OpenRequest request;
  request.set_fsid(user_url_);
  request.set_flags(flags & ~O_CREAT);
  request.set_mode(mode);
  request.set_ino(in->ino);

  OpenReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Open(&ctx, request, &reply);

  int result;
  if (status.ok()) {
    result = 0;
    if (result >= 0) {
      if (out) {
        *out = cache_->CreateFileHandle(in, mode);
      }
    } else {
      goto err;
    }
  } else {
    return -EIO;
  }
 err:
  return result;
}

int Client::Open(Inode *in, int flags, mode_t mode, int uid, int gid, FileHandle **out) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);
  assert(!(flags & O_CREAT));

  FLOG(1) << " ino=" << in->ino
      << ", mode=" << std::oct << mode << std::dec
      << ", uid=" << uid << ", gid=" << gid;

  int result = _Open(in, flags, mode, uid, gid, out);
  return result;
}

int Client::Read(FileHandle *fh, off_t off, size_t size, char *buf) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(fh != nullptr);

  FLOG(1) << " fd=" << fh->fd << ", ino=" << fh->inode->ino
      << ", offset=" << off << ", size=" << size;

  ssize_t result;
  if (0 > (result = fh->Read(buf, off, size, false))) {
    // TODO(noone) error failed to read file
  }
  return static_cast<int>(result);
}

int Client::Write(FileHandle *fh, const char *buf, off_t off, size_t size) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != fh);

  FLOG(1) << " fd=" << fh->fd << ", ino=" << fh->inode->ino
      << ", offset=" << off << ", size=" << size;

  ssize_t result;
  if (0 > (result = fh->Write(buf, off, size))) {
    // TODO(noone) error failed to write file
  }
  return static_cast<int>(result);
}

int Client::_Create(Inode *parent, const std::string &name, int flags, mode_t mode,
    int uid, int gid, Inode **out, FileHandle **fhp) {
  if (name.length() > NAME_MAX) {
    return -ENAMETOOLONG;
  }

  CreateRequest request;
  request.set_fsid(user_url_);
  request.set_ino(parent->ino);
  request.set_mode(mode);
  request.set_flags(flags | O_CREAT);
  request.set_uid(uid);
  request.set_gid(gid);
  request.set_name(name);

  CreateReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Create(&ctx, request, &reply);

  int result;
  if (status.ok()) {
    result = reply.result();
    if (result < 0) {
      goto err;
    }
    if (fhp) {
      Inode *inp = handle_inodestat(&reply.inode());
      *fhp = cache_->CreateFileHandle(inp, mode);
      *out = inp;
      FLOG(2) << " FileHandle created with fd=" << (*fhp)->fd;
    }
  } else {
    return -EIO;
  }
 err:
  return result;
}

int Client::Create(Inode *parent, const char *name, mode_t mode, int flags, struct stat *attr,
    int uid, int gid, Inode **out, FileHandle **fh) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != parent);

  FLOG(1) << ", parent_ino=" << parent->ino << ", name=" << name
      << ", mode=" << std::oct << mode << std::dec << ", uid=" << uid << ", gid=" << gid;

  Inode *in = nullptr;
  int result = _Lookup(parent, name, uid, gid, &in);

  if ((0 == result) && (flags & O_CREAT) && (flags && O_EXCL)) {
    return -EEXIST;
  }

  if ((-ENOENT == result) && (flags & O_CREAT)) {
    result = _Create(parent, name, flags, mode, uid, gid, &in, fh);
    if (result < 0) {
      goto out;
    }
  }

  if (result < 0) {
    goto out;
  }

  assert(in);
  fill_stat(in, attr);

 out:
  if (result < 0) {
    attr->st_ino = 0;
  }

  if (out) {
    if (in) {
      ll_get_inode(in);
      *out = in;
    }
  }
  return result;
}

int Client::_ReleaseFileHandle(FileHandle *fh) {
  if (fh->inode->dirty)
    _Sync(fh, 0);
  cache_->Close(fh);
  return 0;
}

int Client::Release(FileHandle *fh) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != fh);

  FLOG(1) << ", fd=" << fh->fd << ", ino=" << fh->inode->ino;

  return _ReleaseFileHandle(fh);
}

int Client::_SyncMetadata(FileHandle *fh) {
  FlushRequest request;
  request.set_fsid(user_url_);
  request.set_ino(fh->inode->ino);
  request.set_size(fh->final_size);

  FLOG(1) << " fd=" << fh->fd << ", ino=" << fh->inode->ino
      << ", size=" << fh->final_size;

  FlushReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Flush(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    if (0 == result) {
      handle_inodestat(&reply.inode());
    }
  } else {
    return -EIO;
  }
  return result;
}

int Client::_Sync(FileHandle *fh, int datasync) {
  FLOG(1) << ", fd=" << fh->fd << ", ino=" << fh->inode->ino
      << ", datasync=" << datasync;

  bool is_modified = fh->is_modified;
  int result = fh->DataFlush(false);
  if (!result && is_modified && !datasync) {
    result = _SyncMetadata(fh);
  }
  return result;
}

int Client::Flush(FileHandle *fh) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != fh);

  FLOG(1) << ", fd=" << fh->fd << ", ino=" << fh->inode->ino;
  int result = _Sync(fh, 0);
  return result;
}

int Client::Fsync(FileHandle *fh, int datasync) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != fh);

  FLOG(1) << ", fd=" << fh->fd << ", ino=" << fh->inode->ino
      << ", datasync=" << datasync;

  int result = _Sync(fh, datasync);
  return result;
}

void Client::fill_dirent(struct dirent *de, const char *name, int type, uint64_t ino, loff_t off) {
  std::strncpy(de->d_name, name, 255);
  de->d_name[255] = '\0';
  de->d_ino = ino;
  de->d_off = off;
  de->d_type = type;
	de->d_reclen = sizeof(struct dirent);
}

void Client::fill_stat(Inode *in, struct stat *st) {
  memset(st, 0, sizeof(struct stat));
  st->st_ino = in->ino;
  st->st_mode = in->mode;
  st->st_nlink = in->nlink;
  st->st_uid = in->uid;
  st->st_gid = in->gid;
  st->st_atim.tv_sec = in->atime;
  st->st_mtim.tv_sec = in->mtime;
  st->st_ctim.tv_sec = in->ctime;

  if (in->is_dir()) {
    // noop
  } else if (in->dirty) {
    st->st_size = in->unflushed_size;
  } else {
    st->st_size = in->size;
  }
  FLOG(2) << "   ino=" << in->ino << ", dirty=" << in->dirty
      << ", nlink=" << st->st_nlink << ", mode="
      << std::oct << st->st_mode << std::dec;
}

int Client::Getattr(Inode *in, struct stat *attr, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);

  FLOG(1) << " ino=" << in->ino << ", uid=" << uid << ", gid=" << gid;

  int result;

  GetattrRequest request;
  request.set_fsid(user_url_);

  request.set_ino(in->ino);

  GetattrReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Getattr(&ctx, request, &reply);

  if (status.ok()) {
    Inode *inp = handle_inodestat(&reply.inode());
    fill_stat(inp, attr);
    // TODO return approriate result
    result = 0;
  } else {
    return -EIO;
  }
  return result;
}

int Client::_Setattr(Inode *in, struct stat *attr, int mask, int uid, int gid, Inode **out) {
  SetattrRequest request;
  request.set_fsid(user_url_);
  auto inode = request.mutable_inode();
  inode->set_ino(in->ino);

  if (mask & OIOFS_SETATTR_MODE) {
    inode->set_mode(attr->st_mode);
  }
  if (mask & OIOFS_SETATTR_UID) {
    inode->set_uid(attr->st_uid);
  }
  if (mask & OIOFS_SETATTR_GID) {
    inode->set_gid(attr->st_gid);
  }
  if (mask & OIOFS_SETATTR_ATIME) {
    inode->set_atime(attr->st_atime);
  }
  if (mask & OIOFS_SETATTR_MTIME) {
    inode->set_mtime(attr->st_mtime);
  }
  if (mask & OIOFS_SETATTR_SIZE) {
    inode->set_size(attr->st_size);
  }
  request.set_mask(mask);

  SetattrReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Setattr(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    if (0 == result) {
      *out = handle_inodestat(&reply.inode());
    }
  } else {
    return -EIO;
  }
  return result;
}

int Client::Setattr(Inode *in, struct stat *attr, int mask, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);

  FLOG(1) << " ino=" << in->ino << ", mask=" << mask
      << ", uid=" << uid << ", gid=" << gid;

  Inode *out = nullptr;
  int result = _Setattr(in, attr, mask, uid, gid, &out);
  if (0 == result) {
    fill_stat(out, attr);
  }
  return result;
}

int Client::_Mkdir(Inode *dir, const std::string& name, mode_t mode,
    int uid, int gid, Inode **out) {
  if (name.length() > NAME_MAX) {
    return -ENAMETOOLONG;
  }

  MkdirRequest request;
  request.set_fsid(user_url_);
  request.set_ino(dir->ino);
  request.set_mode(mode);
  request.set_uid(uid);
  request.set_gid(gid);
  request.set_name(name);

  MkdirReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Mkdir(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    *out = handle_inodestat(&reply.inode());
  } else {
    return -EIO;
  }
  return result;
}

int Client::Mkdir(Inode *dir, const std::string& name, mode_t mode, struct stat *attr,
    int uid, int gid, Inode **out) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != dir);

  FLOG(1) << " parent_ino=" << dir->ino << ", name=" << name
      << ", uid=" << uid << ", gid=" << gid
      << ", mode=" << std::oct << mode << std::dec;

  Inode *in = nullptr;
  int result = _Mkdir(dir, name, mode, uid, gid, &in);
  if (0 == result) {
    fill_stat(in, attr);
    ll_get_inode(in);
  }
  *out = in;
  return result;
}

int Client::_Rmdir(Inode *in, const std::string &name, int uid, int gid) {
  RmdirRequest request;
  request.set_fsid(user_url_);
  request.set_ino(in->ino);
  request.set_name(name);

  RmdirReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Rmdir(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
  } else {
    return -EIO;
  }
  return result;
}

int Client::Rmdir(Inode *in, const std::string& name, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);

  return _Rmdir(in, name, uid, gid);
}

int Client::Mknod(Inode *parent, const char *name, mode_t mode, dev_t rdev, struct stat *attr,
    int uid, int gid, Inode **out) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != parent);

  FLOG(1) << " parent_ino=" << parent->ino << ", name=" << name
      << ", mode=" << std::oct << mode << std::dec
      << ", uid=" << uid << ", gid=" << gid;

  int result = 0;

  if (std::strlen(name) > NAME_MAX) {
    return -ENAMETOOLONG;
  }

  MknodRequest request;
  request.set_fsid(user_url_);
  request.set_ino(parent->ino);
  request.set_mode(mode);
  request.set_uid(uid);
  request.set_gid(gid);
  request.set_name(name);

  MknodReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Mknod(&ctx, request, &reply);

  if (status.ok()) {
    if (0 == result) {
      Inode *inp = handle_inodestat(&reply.inode());
      fill_stat(inp, attr);
      ll_get_inode(inp);
      *out = inp;
    }
  } else {
    return -EIO;
  }
  return result;
}

int Client::_Readlink(Inode *in, char *buf, size_t buflen) {
  if (!in->is_symlink()) {
    return -EINVAL;
  }

  int result = in->symlink.length();
  if (result > static_cast<int>(buflen)) {
    result = buflen;
  }
  memcpy(buf, in->symlink.c_str(), result + 1);
  return result;
}

int Client::Readlink(Inode *in, char *buf, size_t buflen, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);

  return _Readlink(in, buf, buflen);
}

int Client::Statfs(Inode *in, struct statvfs *attr) {
  std::lock_guard<std::mutex> lock(client_lock_);
  struct oio_sds_usage_s usage = {0};
  struct oio_error_s *err = nullptr;

  assert(nullptr != in);
  attr->f_bsize = 1024;
  attr->f_frsize = 1024;  // should be the same as f_bsize

  attr->f_files = 1024*1024;  // 1M contents per container seems legit
  attr->f_ffree = 1024*1024;
  attr->f_favail = 1024*1024;
//  attr->f_fsid = ;
//  attr->f_flag = ;
  attr->f_namemax = 255;  // should be LIMIT_LENGTH_CONTENTPATH-1

  err = oio_sds_get_usage(oio_client_, oio_url_, &usage);
  if (err) {
    LOG(ERROR) << __FUNCTION__ << " Failed to query container usage: "
        << oio_error_message(err);
    oio_error_pfree(&err);
    // We still continue and report 0 as available size
  }

  attr->f_blocks = usage.quota_bytes / attr->f_bsize;
  if (usage.used_bytes < usage.quota_bytes)
    attr->f_bavail = (usage.quota_bytes - usage.used_bytes) / attr->f_bsize;
  else
    attr->f_bavail = 0;
  attr->f_bfree = attr->f_bavail;

  return 0;
}

int Client::_Opendir(Inode *in, int uid, int gid, dir_result_t **dirpp) {
  if (!in->is_dir()) {
    return -ENOTDIR;
  }
  *dirpp = new dir_result_t(in);
  return 0;
}

int Client::Opendir(Inode *in, int uid, int gid, dir_result_t **dirpp) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);
  int err = _Opendir(in, uid, gid, dirpp);

  return err;
}

void Client::_Releasedir(dir_result_t *dirp) {
  _drop_dirp_buffer(dirp);
  delete dirp;
}

int Client::Releasedir(dir_result_t *dirp) {
  std::lock_guard<std::mutex> lock(client_lock_);

  _Releasedir(dirp);

  return 0;
}

int Client::_Rename(Inode *parent, const std::string &name, Inode *newparent,
    const std::string &newname, int uid, int gid) {
  RenameRequest request;
  request.set_fsid(user_url_);
  request.set_ino(parent->ino);
  request.set_name(name);
  request.set_new_ino(newparent->ino);
  request.set_new_name(newname);

  RenameReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Rename(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    // TODO remove file in oio?
  } else {
    return -EIO;
  }
  return result;
}

int Client::Rename(Inode *parent, const std::string &name, Inode *newparent,
    const std::string &newname, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != parent);

  FLOG(1) << " parent_ino=" << parent->ino << ", newparent_ino="
      << newparent->ino << ", name=" << name << ", new_name=" << newname
      << ", uid=" << uid << ", gid=" << gid;

  return _Rename(parent, name, newparent, newname, uid, gid);
}

int Client::_Link(Inode *in, Inode *newparent, const std::string &name,
    int uid, int gid, Inode **out) {
  if (name.length() > NAME_MAX) {
    return -ENAMETOOLONG;
  }

  LinkRequest request;
  request.set_fsid(user_url_);
  request.set_ino(in->ino);
  request.set_name(name);
  request.set_new_ino(newparent->ino);

  LinkReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Link(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    if (0 == result) {
      *out = handle_inodestat(&reply.inode());
    }
  } else {
    return -EIO;
  }
  return result;
}

int Client::Link(Inode *in, Inode *newparent, const std::string &name,
    struct stat *attr, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);

  FLOG(1) << " ino=" << in->ino << ", parent_ino=" << newparent->ino
      << ", name=" << name << ", uid=" << uid << ", gid=" << gid;

  Inode *inp = nullptr;
  int result = _Link(in, newparent, name, uid, gid, &inp);
  if (0 == result) {
    fill_stat(inp, attr);
    ll_get_inode(inp);
  }
  return result;
}

int Client::_Unlink(Inode *in, const std::string &name, int uid, int gid) {
  UnlinkRequest request;
  request.set_fsid(user_url_);
  request.set_ino(in->ino);
  request.set_name(name);

  UnlinkReply reply;
  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Unlink(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    // TODO remove file on oio?
  } else {
    return -EIO;
  }
  return result;
}

int Client::Unlink(Inode *in, const std::string &name, int uid, int gid) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != in);

  FLOG(1) << " ino=" << in->ino << ", uid=" << uid << ", gid=" << gid;

  return _Unlink(in, name, uid, gid);
}

int Client::_Symlink(Inode *dir, const std::string &name, const std::string &target,
    int uid, int gid, Inode **out) {
  if (name.length() > NAME_MAX) {
    return -ENAMETOOLONG;
  }

  SymlinkRequest request;
  request.set_fsid(user_url_);
  request.set_ino(dir->ino);
  request.set_uid(uid);
  request.set_gid(gid);
  request.set_name(name);
  request.set_target(target);

  SymlinkReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Symlink(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    *out = handle_inodestat(&reply.inode());
  } else {
    return -EIO;
  }
  return result;
}

int Client::Symlink(Inode *dir, const std::string &name, const std::string &target,
    struct stat *attr, int uid, int gid, Inode **out) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != dir);

  Inode *in = nullptr;
  int result = _Symlink(dir, name, target, uid, gid, &in);
  if (0 == result) {
    fill_stat(in, attr);
    ll_get_inode(in);
  }
  *out = in;
  return result;
}

int Client::_Readdir(dir_result_t *dirp) {
  ReaddirRequest request;
  request.set_fsid(user_url_);
  request.set_ino(dirp->inode->ino);
  request.set_offset(dirp->offset);

  ReaddirReply reply;

  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  grpc::Status status = stub_->Readdir(&ctx, request, &reply);

  int result = 0;
  if (status.ok()) {
    result = reply.result();
    if (0 == result) {
      _drop_dirp_buffer(dirp);
      dirp->buffer = new std::vector<std::pair<std::string, ino_t>>;
      auto dirents = reply.dirents();
      for (auto iter = dirents.begin(); iter != dirents.end(); ++iter) {
        dirp->buffer->push_back(std::pair<std::string, ino_t>((*iter).name(), (*iter).ino()));
      }
    } else {
      dirp->set_end();
    }
  } else {
    dirp->set_end();
    return -EIO;
  }
  return result;
}

struct getdents_result {
  char *buf;
  int buflen;
  int pos;
};

static int readdir_getdents_cb(void *p, struct dirent *de, struct stat *st, int stmask, off_t off) {
  struct getdents_result *gr = static_cast<getdents_result*>(p);
  int dlen = sizeof(*de);
  if (gr->pos + dlen > gr->buflen) {
    return -1;
  }
  memcpy(gr->buf + gr->pos, de, sizeof(*de));
  gr->pos += dlen;
  return 0;
}

int Client::Getdents(dir_result_t *dirp, char *buf, int buflen) {
  struct getdents_result gr;
  gr.buf = buf;
  gr.buflen = buflen;
  gr.pos = 0;

  int result = ReaddirRCb(dirp, readdir_getdents_cb, &gr);

  if (result < 0) {
    if (-1 == result) {
      if (gr.pos) {
        return gr.pos;
      }
      return -ERANGE;
    } else {
      return result;
    }
  }
  return gr.pos;
}

struct single_result {
  struct dirent *de;
  struct stat *st;
  int *stmask;
  bool full;
};

static int readdir_single_cb(void *p, struct dirent *de, struct stat *st, int stmask, off_t off) {
  single_result *sr = static_cast<single_result*>(p);
  if (sr->full) {
    return -1;
  }
  *sr->de = *de;
  if (sr->st) {
    *sr->st = *st;
  }
  if (sr->stmask) {
    *sr->stmask = stmask;
  }
  sr->full = true;
  return 1;
}

void Client::Rewinddir(dir_result_t *dirp) {
  std::lock_guard<std::mutex> lock(client_lock_);
  _drop_dirp_buffer(dirp);
  dirp->reset();
}

loff_t Client::Telldir(dir_result_t *dirp) {
  return dirp->offset;
}

struct dirent* Client::Readdir(dir_result_t *dirp) {
  int result;
  static int stmask;
  static struct dirent de;
  static struct stat st;
  struct single_result sr;
  sr.de = &de;
  sr.st = &st;
  sr.stmask = &stmask;
  sr.full = false;

  result = ReaddirRCb(dirp, readdir_single_cb, &sr);
  if (result < -1) {
    return nullptr;
  }
  if (sr.full) {
    return &de;
  }
  return nullptr;
}

int Client::ReaddirRCb(dir_result_t *dirp, add_dirent_cb_t cb, void *p) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != dirp);

  struct dirent de = {0};
  struct stat st = {0};

  if (dirp->at_end()) {
    return 0;
  }

  if (nullptr == dirp->buffer) {
    int err = _Readdir(dirp);
    if (err) {
      return err;
    }
  }

  while (static_cast<size_t>(dirp->offset)< dirp->buffer->size()) {
    // fill dirent name ino
    auto entry = dirp->buffer->at(dirp->offset);

    int stmask = 0;
    // support d_type ?
    fill_dirent(&de, entry.first.c_str(), DT_UNKNOWN, entry.second, dirp->offset + 1);
    int err = cb(p, &de, &st, stmask, dirp->offset + 1);
    if (err < 0) {
      return err;
    }
    dirp->offset++;
    if (err > 0) {
      return err;
    }
  }
  dirp->set_end();
  return 0;
}

int Client::Fallocate(FileHandle *fh, int mode, loff_t offset, loff_t length) {
  std::lock_guard<std::mutex> lock(client_lock_);

  assert(nullptr != fh);

  return -EOPNOTSUPP;
}

void Client::_drop_dirp_buffer(dir_result_t *dirp) {
  if (dirp->buffer) {
    delete dirp->buffer;
    dirp->buffer = nullptr;
  }
}

void Client::Seekdir(dir_result_t *dirp, loff_t off) {
  std::lock_guard<std::mutex> lock(client_lock_);

  if (0 == off) {
    _drop_dirp_buffer(dirp);
    dirp->reset();
  }

  if (off > dirp->offset) {
    dirp->release_count--;
  }
}

int Client::_Mkfs() {
  std::lock_guard<std::mutex> lock(client_lock_);

  int rc = 0;
  if (!connected_ && (rc = _Connect()) != 0) {
    return rc;
  }

  struct oio_error_s *err = oio_sds_create(oio_client_, oio_url_);
  if (err != nullptr) {
    rc = oio_error_code(err);
    LOG(ERROR) << "Failed to create container for " << user_url_
        << ": (" << rc << ") " << oio_error_message(err);
    oio_error_pfree(&err);
    return rc;
  }

  MkfsRequest request;
  request.set_fsid(user_url_);

  MkfsReply reply;
  grpc::ClientContext ctx;
  _PrepareContext(&ctx);
  auto status = stub_->Mkfs(&ctx, request, &reply);

  if (!status.ok()) {
    LOG(ERROR) << "Failed to create filesystem for " << user_url_ << ": ("
        << status.error_code() << ") " << status.error_message();
    rc = status.error_code();
  } else if (reply.result() != 0) {
    std::string message = reply.error();
    LOG(ERROR) << "Failed to create filesystem for " << user_url_ << ": ("
        << reply.result() << ") "
        << (message.empty()? message.c_str() : "no error message");
    rc = reply.result();
  }

  return rc;
}

int mkfs(const std::string &user_url, std::string *srv_addr_p) {
  int rc = 0;
  std::string srv_addr;
  struct oio_url_s *url = oio_url_init(user_url.c_str());

  if (srv_addr_p != nullptr) {
    // Caller provided the addr of the oiofs-server to use
    srv_addr = *srv_addr_p;
  } else {
    // Ask conscience to link an oiofs-server to the container
    GError *gerr = nullptr;
    gchar **services = nullptr;
    struct oio_directory_s *dir = nullptr;
    dir = oio_directory__create_proxy(oio_url_get(url, OIOURL_NS));
    gerr = oio_directory__link(dir, url, OIOFS_SRV_TYPE, true, &services);
    oio_directory__destroy(dir);
    if (gerr != nullptr) {
      LOG(ERROR) << "Failed to prepare filesystem: (" << gerr->code
          << ") " << gerr->message;
      g_clear_error(&gerr);
      rc = EXIT_FAILURE;
    } else {
      // There should be only one
      std::string srv_entry = std::string(services[0]);
      parse_service_addr(srv_entry, &srv_addr);
      g_strfreev(services);
    }
  }

  if (rc == 0) {
    oiofs::Client client(user_url, srv_addr);
    rc = client._Mkfs();
  }

  oio_url_pclean(&url);
  return rc;
}

struct oio_url_s* Client::url_from_ino(ino_t ino_no) {
  struct oio_url_s *new_url = oio_url_dup(oio_url_);
  char in_path[16];
  snprintf(in_path, sizeof(in_path), "%lu", ino_no);
  oio_url_set(new_url, OIOURL_PATH, in_path);
  return new_url;
}

}  // namespace oiofs
