/*
 *
 * Copyright 2015, OpenIO
 *
 */    

#ifndef OIOFS_INTERNAL_META_CLIENT_H_
#define OIOFS_INTERNAL_META_CLIENT_H_

#include <glog/logging.h>
#include <grpc++/grpc++.h>
#include <oio_core.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <map>
#include <chrono>
#include <vector>
#include <string>
#include <utility>
#include <iostream>
#include "include/oiofs.h"
#include "meta.pb.h"
#include "meta.grpc.pb.h"
#include "src/inode.h"
#include "src/oiofs_defaults.h"
#include "src/file_cache.h"

namespace oiofs {

typedef int (*add_dirent_cb_t)(void *p, struct dirent *de, struct stat *st, int mask, off_t off);

struct dir_result_t {
  explicit dir_result_t(Inode *in);

  Inode *inode;
  int64_t offset;
  uint64_t next_offset;
  uint64_t release_count;
  bool end;
  std::vector<std::pair<std::string, ino_t>> *buffer;
  std::string last_name;

  bool at_end() { return end; }
  void set_end() { end = true; }
  void reset() {
    last_name.clear();
    offset = 0;
    delete buffer;
    buffer = nullptr;
  }
};

class Client {
 public:
  Client() : Client(OIOFS_DEFAULT_USER_URL,
      OIOFS_DEFAULT_BIND_ADDR,
      OIOFS_DEFAULT_CACHE_PATH) {}
  explicit Client(
      std::string user_url,
      std::string srv_addr = "",  // If empty, get from directory
      std::string cache_dir = OIOFS_DEFAULT_CACHE_PATH,
      std::chrono::seconds timeout = std::chrono::seconds(30));
  ~Client();

  int Init();
  int Mount();
  void Unmount();
  void Shutdown();
  int Lookup(Inode *parent, const std::string& path, struct stat *attr,
      int uid, int gid, Inode **out);
  int GetInode(ino_t ino, Inode **out);
  bool PutInode(Inode *inode);
  bool ForgetInode(Inode *inode, int count);
  int Getattr(Inode *inode, struct stat *attr, int uid, int gid);
  int Setattr(Inode *inode, struct stat *attr, int mask, int uid, int gid);
  int Readlink(Inode *inode, char *buf, size_t bufsize, int uid, int gid);
  int Mknod(Inode *inode, const char *name, mode_t mode, dev_t rdev,
      struct stat *attr, int uid, int gid, Inode **out);
  int Mkdir(Inode *inode, const std::string& name, mode_t mode, struct stat *stat,
      int uid, int gid, Inode **out);
  int Unlink(Inode *inode, const std::string& name, int uid, int gid);
  int Rmdir(Inode *inode, const std::string& name, int uid, int gid);
  int Symlink(Inode *inode, const std::string& name, const std::string& existing,
      struct stat *attr, int uid, int gid, Inode **out);
  int Rename(Inode *inode, const std::string& name, Inode *new_inode, const std::string& new_name,
      int uid, int gid);
  int Link(Inode *inode, Inode *new_inode, const std::string& new_name, struct stat *attr,
      int uid, int gid);
  int Open(Inode *inode, int flags, mode_t mode, int uid, int gid, FileHandle **out);
  int Read(FileHandle *fh, off_t off, size_t size, char *buf);
  int Write(FileHandle *fh, const char *buf, off_t off, size_t size);
  int Create(Inode *parent, const char *name, mode_t mode, int flags, struct stat *attr,
      int uid, int gid, Inode **out, FileHandle **fh);
  int Flush(FileHandle *fh);
  /** Actually synchronize data to object storage.
   *  `datasync` is currently not used since attributes changes are all
   *  done synchronously. */
  int Fsync(FileHandle *fh, int datasync);
  int Fallocate(FileHandle *fh, int mode, off_t off, off_t length);
  int Release(FileHandle *fh);
  int Opendir(Inode *in, int uid, int gid, dir_result_t **out);
  int Getdents(dir_result_t *dirp, char *buf, int buflen);
  void Rewinddir(dir_result_t *dirp);
  loff_t Telldir(dir_result_t *dirp);
  void Seekdir(dir_result_t *dirp, loff_t off);
  struct dirent* Readdir(dir_result_t *dirp);
  int ReaddirRCb(dir_result_t *dirp, add_dirent_cb_t cb, void *p);
  int Releasedir(dir_result_t *dirp);
  int Fsyncdir();
  int Access();
  int Statfs(Inode *inode, struct statvfs *attr);

  /** Create the filesystem. The name is taken from class constructor */
  int _Mkfs();

 private:
  bool connected_;
  bool mounted_;
  std::string user_url_;
  struct oio_url_s *oio_url_;
  std::string server_addr_;
  std::shared_ptr<grpc::Channel> channel_;
  std::unique_ptr<MetaService::Stub> stub_ = nullptr;
  std::string cache_dir_;
  std::unique_ptr<FileCache> cache_;
  struct oio_sds_s *oio_client_;
  Inode* root_inode_;
  std::map<ino_t, Inode*> inode_map_;
  std::mutex client_lock_;
  std::chrono::seconds timeout_;

  /** Get server_addr_ from directory if empty,
   *  then create channel_ and stub_ and set connected_ to true. */
  int _Connect();
  void _PrepareContext(grpc::ClientContext *context);

  Inode* handle_inodestat(const InodeStat *inode_stat);
  void fill_stat(Inode *in, struct stat *st);
  void fill_dirent(struct dirent *d, const char *name, int type, uint64_t ino, loff_t off);
  int _PerformLookup(Inode *parent, const std::string &name, int uid, int gid,
      Inode **out);
  int _Lookup(Inode *parent, const std::string &name, int uid, int gid, Inode **out);
  int _Setattr(Inode *in, struct stat *attr, int mask, int uid, int gid, Inode **out);
  int _Open(Inode *in, int flags, mode_t mode, int uid, int gid, FileHandle **out);
  int _Create(Inode *in, const std::string &name, int flags, mode_t mode,
      int uid, int gid, Inode **out, FileHandle **fhp);
  int _Symlink(Inode *dir, const std::string &name, const std::string &target,
      int uid, int gid, Inode **out);
  int _Mkdir(Inode *dir, const std::string &name, mode_t mode, int uid, int gid, Inode **out);
  int _Rmdir(Inode *in, const std::string &name, int uid, int gid);
  int _Rename(Inode *inode, const std::string &name, Inode *new_inode, const std::string& new_name,
      int uid, int gid);
  int _Readlink(Inode *in, char *buf, size_t size);
  int _Link(Inode *in, Inode *newparent, const std::string &name, int uid, int gid, Inode **out);
  int _Unlink(Inode *in, const std::string &name, int uid, int gid);
  void _drop_dirp_buffer(dir_result_t *dirp);
  int _Opendir(Inode *in, int uid, int gid, dir_result_t **dirpp);
  int _Readdir(dir_result_t *dirp);
  /** Send metadata (actually only size) to metadata backend. */
  int _SyncMetadata(FileHandle *fh);
  /** Send data and metadata to backend. */
  int _Sync(FileHandle *fh, int datasync);
  void _FlushAll(Inode *in);
  void _Releasedir(dir_result_t *dirp);
  int _ReleaseFileHandle(FileHandle *fh);
  int _PerformForget(Inode *in);
  void put_filehandle(FileHandle *fh);
  void put_inode(Inode *in, int count);
  int ll_put_inode(Inode *in, int count);
  void get_inode(Inode *in);
  void ll_get_inode(Inode *in);
  struct oio_url_s *url_from_ino(ino_t ino_no);
};

int mkfs(const std::string &user_url, std::string *srv_addr = nullptr);
}  // namespace oiofs

#endif  // OIOFS_INTERNAL_META_CLIENT_H_
