/*
 *
 * Copyright 2015, OpenIO
 *
 */

#include <oio_core.h>

#include <getopt.h>
#include <grpc++/grpc++.h>
#include <cstdlib>
#include <iostream>
#include <string>
#include <thread>
#include <chrono>
#include "src/meta/server_builder.h"
#include "src/oiofs_defaults.h"

#define OIOFS_STAT_FLOAT_LEN 16
static char bind_addr[256] = OIOFS_DEFAULT_BIND_ADDR;
// This is named "host" but must contain an IP address
static char redis_host[256] = OIOFS_DEFAULT_REDIS_HOST;
static int redis_port = OIOFS_DEFAULT_REDIS_PORT;
static char master_name[128] = OIOFS_DEFAULT_REDIS_MASTER_NAME;
static char sentinels[1024] = {0};

static bool running = true;
static char *ns_name = nullptr;
static int registration_interval = 5;
static std::thread *register_thread = nullptr;
static std::unique_ptr<oiofs::Server> server = nullptr;

void print_help(const char *prog_name) {
  printf("Usage: %s [OPTIONS] NAMESPACE\n", prog_name);
  printf("\nOptions:\n");
  printf("\t-b, --bind <host:port>   Specify the host and port where to listen"
      "\n\t                        (\"" OIOFS_DEFAULT_BIND_ADDR "\" by default)\n");
  printf("\t-d, --debug              Enable debug traces (not implemented)\n");
  printf("\t-g, --register-interval <seconds> Interval between conscience "
      "registrations (5s by default)\n");
  printf("\t-h, --help               Print this help\n");
  printf("\t-m, --master-name <name> Specify the name of master Redis when using Sentinels\n");
  printf("\t-p, --redis-port <port>  Set the port of the Redis backend\n");
  printf("\t-r, --redis-host <addr>  Set the IP addr of the Redis backend"
      "\n\t                        (\"" OIOFS_DEFAULT_REDIS_HOST "\" by default)\n");
  printf("\t-s, --sentinels <IP:PORT,IP:PORT,...> Specify the list of Redis sentinels\n");
}

static void self_register() {
  auto last_register = std::chrono::steady_clock::now();
  struct oio_cs_registration_s registration = {0};
  registration.url = bind_addr;
  // TODO(fvennetier): get a custom id from command line
  registration.id = bind_addr;
  char **tags = reinterpret_cast<char**>(calloc(5, sizeof(char*)));
  tags[0] = strdup("tag.up");
  tags[1] = strdup("true");
  tags[2] = strdup("stat.cpu");
  tags[3] = (char*) malloc(OIOFS_STAT_FLOAT_LEN);
  // tags[4] = nullptr;  // implicit
  registration.kv_tags = tags;

  struct oio_cs_client_s *cs_client = oio_cs_client__create_proxied(ns_name);
  while (running) {
    auto now = std::chrono::steady_clock::now();
    auto diff = std::chrono::duration_cast<std::chrono::seconds>(
        now - last_register);
    if (diff > std::chrono::seconds(registration_interval)) {
      double cpu_idle = 100.0 * oio_sys_cpu_idle();
      snprintf(tags[3], OIOFS_STAT_FLOAT_LEN, "%lf", cpu_idle);
      oio_cs_client__register_service(cs_client, OIOFS_SRV_TYPE, &registration);
      last_register = now;
    }
    std::this_thread::sleep_for(std::chrono::seconds(1));
  }

  printf("De-registering from conscience...\n");
  char *tag_up = tags[1];
  tags[1] = strdup("false");
  free(tag_up);
  oio_cs_client__register_service(cs_client, OIOFS_SRV_TYPE, &registration);

  oio_cs_client__destroy(cs_client);
  for (char **ptr = tags; ptr && *ptr; ptr++) {
    free(*ptr);
    *ptr = nullptr;
  }
  free(tags);
}

static void sighandler_exit(int s) {
  running = false;
  server->Shutdown();
}

int main(int argc, char *argv[]) {
  int c;

  static struct option long_options[] = {
    {"help", no_argument, nullptr, 'h'},
    {"verbose", no_argument, nullptr, 'v'},
    {"redis-host", 1, nullptr, 'r'},
    {"redis-port", 1, nullptr, 'p'},
    {"bind", 1, nullptr, 'b'},
    {"register-interval", 1, nullptr, 'g'},
    {"master-name", 1, nullptr, 'm'},
    {"sentinels", 1, nullptr, 's'},
    {0, 0, 0, 0}
  };

  google::InitGoogleLogging(argv[0]);

  while (1) {
    int tmp_int = 0;
    int option_index = 0;

    c = getopt_long(argc, argv, "hvb:g:m:p:r:s:", long_options, &option_index);
    if (-1 == c) {
      break;
    }

    switch (c) {
    case 'b':
      strncpy(bind_addr, optarg, sizeof(bind_addr));
      break;
    case 'g':
      tmp_int = atoi(optarg);
      if (tmp_int == 0) {
        fprintf(stderr, "Invalid register interval: %s", optarg);
      } else {
        registration_interval = tmp_int;
      }
      break;
    case 'h':
      print_help(argv[0]);
      exit(EXIT_SUCCESS);
    case 'p':
      tmp_int = atoi(optarg);
      if (tmp_int <= 0 || tmp_int > 65535) {
        fprintf(stderr, "Invalid Redis port: %s", optarg);
      } else {
        redis_port = tmp_int;
      }
      break;
    case 'r':
      strncpy(redis_host, optarg, sizeof(redis_host)-1);
      break;
    case 'm':
      strncpy(master_name, optarg, sizeof(master_name)-1);
      break;
    case 's':
      strncpy(sentinels, optarg, sizeof(sentinels)-1);
      break;
    case 'v':
      // Same as seting GLOG_v environment variable
      fLI::FLAGS_v++;
      break;
    case '?':
      break;
    default:
      printf("?? getopt returned character code 0%o ??\n", c);
    }
  }

  if (optind >= argc) {
    fprintf(stderr, "Missing argument: NAMESPACE\n");
    print_help(argv[0]);
    exit(EXIT_FAILURE);
  }
  ns_name = argv[optind];

  oiofs::ServerBuilder builder;

  std::string addr(bind_addr);
  builder.AddListeningPort(addr);

  std::string redis_host_str(redis_host);
  builder.SetRedisHost(redis_host_str);
  builder.SetRedisPort(redis_port);
  if (sentinels[0]) {
    std::string sentinels_str(sentinels);
    builder.AddSentinels(sentinels_str);
  }
  std::string master_name_str(master_name);
  builder.SetSentinelMasterName(master_name_str);

  server = builder.BuildAndStart();
  signal(SIGTERM, sighandler_exit);
  signal(SIGINT, sighandler_exit);
  signal(SIGQUIT, sighandler_exit);
  signal(SIGHUP, sighandler_exit);
  if (registration_interval > 0) {
    register_thread = new std::thread(self_register);
  }
  server->Wait();
  printf("Shutting down...\n");

  running = false;
  if (register_thread != nullptr) {
    register_thread->join();
    delete register_thread;
  }

  return 0;
}
