/*
 *
 * Copyright 2015, OpenIO SAS
 *
 */

#include "src/meta/service.h"
#include <string.h>
#include <sys/stat.h>
#include <stdint.h>
#include <grpc++/grpc++.h>
#include <hiredis/hiredis.h>
#include <iostream>
#include <sstream>
#include <cstring>
#include <string>
#include "include/oiofs.h"

#ifndef FLOG
# define FLOG(level) VLOG(level) << __FUNCTION__
#endif

namespace oiofs {

static std::string kDotdot = "..";

grpc::Status MetaServiceImpl::Lookup(grpc::ServerContext* context, const LookupRequest* request,
    LookupReply* reply) {

  auto fsid = request->fsid();
  auto ino = request->ino();
  auto name = request->name();
  InodeStat *inode = nullptr;
  int err = 0;

  err = backend_->LookupInodeStat(fsid, ino, name, &inode);
  reply->set_result(err);
  if (nullptr != inode) {
    reply->set_allocated_inode(inode);
  }
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Getattr(grpc::ServerContext *context, const GetattrRequest *request,
    GetattrReply *reply) {
  auto fsid = request->fsid();
  InodeStat *inode = nullptr;
  auto ino = request->ino();

  FLOG(1) <<" fsid=" << fsid << ", ino=" << ino;

  int result = backend_->GetInodeStat(fsid, ino, &inode);
  reply->set_result(result);
  if (nullptr != inode) {
    reply->set_allocated_inode(inode);
  }
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Setattr(grpc::ServerContext *context, const SetattrRequest *request,
    SetattrReply *reply) {
  auto fsid = request->fsid();
  auto inode = request->inode();
  auto ino = inode.ino();
  auto mask = request->mask();
  int result = 0;
  InodeStat *new_inode = nullptr;
  int err = 0;
  auto now = std::time(nullptr);

  err = backend_->GetInodeStat(fsid, ino, &new_inode);
  if (err) {
    goto out;
  }

  if (mask & OIOFS_SETATTR_MODE) {
    new_inode->set_mode((new_inode->mode() & ~07777) | (inode.mode() & 07777));
  }
  if (mask & OIOFS_SETATTR_UID) {
    new_inode->set_uid(inode.uid());
  }
  if (mask & OIOFS_SETATTR_GID) {
    new_inode->set_gid(inode.gid());
  }
  if (mask & OIOFS_SETATTR_MTIME) {
    new_inode->set_mtime(inode.mtime());
  }
  if (mask & OIOFS_SETATTR_ATIME) {
    new_inode->set_atime(inode.atime());
  }
  if (mask & OIOFS_SETATTR_MTIME_NOW) {
    new_inode->set_mtime(now);
  }
  if (mask & OIOFS_SETATTR_ATIME_NOW) {
    new_inode->set_atime(now);
  }
  if (mask & OIOFS_SETATTR_SIZE) {
    new_inode->set_size(inode.size());
  }
  new_inode->set_ctime(now);

  FLOG(1) <<" fsid=" << fsid << ", ino=" << ino
      << ", mask=" << mask;

  if (ino == OIOFS_INO_ROOT) {
    // root inode special case
  } else {
    result = backend_->SetInodeStat(fsid, ino, new_inode);
    if (0 != result) {
      goto out;
    }
  }

  if (nullptr != new_inode) {
    reply->set_allocated_inode(new_inode);
  }
 out:
  reply->set_result(result);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Create(grpc::ServerContext* context, const CreateRequest* request,
    CreateReply *reply) {
  int err = 0;
  auto fsid = request->fsid();
  auto ino = request->ino();
  InodeStat *new_inode = nullptr;
  mode_t mode = request->mode();
  uid_t uid = request->uid();
  gid_t gid = request->gid();

  if (0 == (mode & S_IFMT)) {
    mode |= S_IFREG;
  }

  FLOG(1) <<" fsid=" << fsid << ", ino=" << ino
      << ", name=" << request->name()
      << ", uid=" << uid << ", gid=" << gid << ", mode=" << std::oct << mode
      << std::dec;

  err = backend_->AllocateInode(fsid, mode, uid, gid, &new_inode);
  if (err) {
    goto out;
  }
  err = backend_->AddLink(fsid, ino, request->name(), new_inode->ino());
  if (err) {
    // TODO (noone) remove inode
    goto out;
  }
  backend_->IncrNlink(fsid, new_inode->ino(), 1);
  if (nullptr != new_inode) {
    reply->set_allocated_inode(new_inode);
  }
 out:
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Open(grpc::ServerContext* context, const OpenRequest* request,
    OpenReply *reply) {
  // check parent object access
  // check object access
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Mkdir(grpc::ServerContext *context, const MkdirRequest *request,
    MkdirReply *reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  mode_t mode = request->mode();
  mode &= ~S_IFMT;
  mode |= S_IFDIR;

  uid_t uid = request->uid();
  gid_t gid = request->gid();

  InodeStat *new_inode = nullptr;
  int err = 0;

  err = backend_->AllocateInode(fsid, mode, uid, gid, &new_inode);
  if (err) {
    goto out;
  }

  err = backend_->AddDir(fsid, ino, request->name(), new_inode->ino());
  if (err) {
    goto out;
  }

  err = backend_->AddLink(fsid, ino, request->name(), new_inode->ino());
  if (err) {
    goto out;
  }

  backend_->IncrNlink(fsid, new_inode->ino(), 1);

  if (nullptr != new_inode) {
    reply->set_allocated_inode(new_inode);
  }

 out:
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Mknod(grpc::ServerContext *context, const MknodRequest *request,
    MknodReply *reply) {
  auto fsid = request->fsid();
  mode_t mode = request->mode();
  if (0 == (mode & S_IFMT)) {
    mode |= S_IFREG;
  }

  uid_t uid = request->uid();
  gid_t gid = request->gid();

  FLOG(1) <<" fsid=" << fsid << ", parent_ino=" << request->ino()
    << ", name=" << request->name()
    << ", mode=" << std::oct << mode << std::dec
    << ", uid=" << uid << ", gid=" << gid;

  InodeStat *new_inode = nullptr;
  int err = 0;
  err = backend_->AllocateInode(fsid, mode, uid, gid, &new_inode);

  if (0 == err) {
    err = backend_->AddLink(fsid, request->ino(), request->name(), new_inode->ino());
    if (err) {
      goto out;
    }

    backend_->IncrNlink(fsid, new_inode->ino(), 1);
    if (nullptr != new_inode) {
      reply->set_allocated_inode(new_inode);
    }
  }

 out:
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Symlink(grpc::ServerContext *context, const SymlinkRequest* request,
    SymlinkReply* reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  auto target = request->target();
  mode_t mode = S_IFLNK | 0777;
  uid_t uid = request->uid();
  gid_t gid = request->gid();

  InodeStat *new_inode = nullptr;
  int err = 0;
  err = backend_->AllocateInode(fsid, mode, uid, gid, &new_inode);

  if (0 == err) {
    err = backend_->AddLink(fsid, ino, request->name(), new_inode->ino());
    if (err) {
      goto out;
    }
    err = backend_->SetSymlink(fsid, new_inode->ino(), target);
    if (err) {
      goto out;
    }

    backend_->IncrNlink(fsid, new_inode->ino(), 1);

    new_inode->set_symlink(target);
    if (nullptr != new_inode) {
      reply->set_allocated_inode(new_inode);
    }
  }
 out:
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Link(grpc::ServerContext* context, const LinkRequest* request,
    LinkReply* reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  auto name = request->name();
  auto new_ino = request->new_ino();
  int err = 0;

  FLOG(1) <<" fsid=" << fsid << ", ino=" << ino
    << ", name=" << name << ", parent_ino=" << new_ino;

  InodeStat *inode = nullptr;
  err = backend_->GetInodeStat(fsid, ino, &inode);
  if (err) {
    goto out;
  }
  err = backend_->AddLink(fsid, new_ino, name, ino);
  if (err) {
    goto out;
  }
  backend_->IncrNlink(fsid, ino, 1);
  inode->set_nlink(inode->nlink() + 1);

  if (nullptr != inode) {
    reply->set_allocated_inode(inode);
  }
 out:
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Unlink(grpc::ServerContext* context, const UnlinkRequest* request,
    UnlinkReply* reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  auto name = request->name();
  InodeStat *inode = nullptr;
  int err = 0;

  FLOG(1) <<" fsid=" << fsid << ", ino=" << ino
    << ", name=" << name;

  err = backend_->LookupInodeStat(fsid, ino, name, &inode);
  if (err) {
    goto out;
  }
  err = backend_->DelLink(fsid, ino, name, inode->ino());
  if (err) {
    goto out;
  }
  backend_->IncrNlink(fsid, inode->ino(), -1);
 out:
  reply->set_result(err);
  if (inode != nullptr)
    delete inode;
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Rmdir(grpc::ServerContext *context, const RmdirRequest* request,
    RmdirReply *reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  auto name = request->name();
  InodeStat *inode = nullptr;
  int err = 0;

  err = backend_->LookupInodeStat(fsid, ino, name, &inode);
  if (err) {
    goto out;
  }
  err = backend_->DelDir(fsid, inode->ino());
  if (err) {
    goto out;
  }
  err = backend_->DelLink(fsid, ino, name, inode->ino());
  if (err) {
    goto out;
  }
  backend_->IncrNlink(fsid, inode->ino(), -1);

 out:
  reply->set_result(err);
  if (inode != nullptr)
    delete inode;
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Rename(grpc::ServerContext* context, const RenameRequest* request,
    RenameReply *reply) {
  auto fsid = request->fsid();
  auto old_dir = request->ino();
  auto old_name = request->name();
  auto new_dir = request->new_ino();
  auto new_name = request->new_name();
  InodeStat *old_inode = nullptr;
  InodeStat *new_inode = nullptr;
  InodeStat *dir_inode = nullptr;
  int err = 0;

  FLOG(1) << " fsid=" << fsid
    << ", old_dir=" << old_dir << ", old_name=" << old_name
    << ", new_dir=" << new_dir << ", new_name=" << new_name;

  err = backend_->LookupInodeStat(fsid, old_dir, old_name, &old_inode);
  if (err) {
    goto out;
  }

  if (S_ISDIR(old_inode->mode())) {
    err = backend_->LookupInodeStat(fsid, old_inode->ino(), kDotdot, &dir_inode);
    if (err) {
      goto out;
    }
  }

  err = backend_->LookupInodeStat(fsid, new_dir, new_name, &new_inode);

  if (new_inode) {
    // check not empty
    // . and ..
    if (S_ISDIR(new_inode->mode()) && (new_inode->entries() != 2)) {
      err = -ENOTEMPTY;
    }
    if (err) {
      goto out;
    }
    err = backend_->SetLink(fsid, new_dir, old_name, new_name, old_inode->ino());
    backend_->IncrNlink(fsid, new_inode->ino(), -1);
  } else {
    err = backend_->AddLink(fsid, new_dir, new_name, old_inode->ino());
    if (err) {
      goto out;
    }
    // TODO drop nlink count
  }

  err = backend_->DelLink(fsid, old_dir, old_name, old_inode->ino());

  if (dir_inode) {
    if (old_dir != new_dir) {
      err = backend_->SetLink(fsid, old_inode->ino(), kDotdot, kDotdot, new_dir);
    }
  }

 out:
  if (old_inode)
    delete old_inode;
  if (new_inode)
    delete new_inode;
  if (dir_inode)
    delete dir_inode;
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Readdir(grpc::ServerContext* context, const ReaddirRequest* request,
    ReaddirReply* reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  auto offset = request->offset();
  int err = 0;

  dirent_cb_t f = [&reply](const char *name, ino_t in) {
    auto d = reply->add_dirents();
    d->set_name(name);
    d->set_ino(in);
    return 0;
  };
  err = backend_->Readdir(fsid, ino, offset, f);
  reply->set_result(err);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Forget(grpc::ServerContext *context, const ForgetRequest *request,
    ForgetReply *reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  int result = 0;

  FLOG(1) << " fsid=" << fsid << ", ino=" << ino;

  result = backend_->DeallocateInode(fsid, ino);
  reply->set_result(result);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Flush(grpc::ServerContext *context, const FlushRequest *request,
    FlushReply *reply) {
  auto fsid = request->fsid();
  auto ino = request->ino();
  auto size = request->size();
  int result = 0;
  InodeStat *new_inode = nullptr;
  std::time_t mtime = std::time(nullptr);
  int err = 0;

  // TODO shouldn't we check stuff here ???
  err = backend_->GetInodeStat(fsid, ino, &new_inode);
  if (err) {
    goto out;
  }

  new_inode->set_size(size);
  new_inode->set_mtime(mtime);
  new_inode->set_ctime(mtime);

  FLOG(1) <<" fsid=" << fsid << ", ino=" << ino
      << ", size=" << size;

  result = backend_->SetInodeStat(fsid, ino, new_inode);
  if (0 != result) {
    goto out;
  }

  if (nullptr != new_inode) {
    reply->set_allocated_inode(new_inode);
  }
 out:
  reply->set_result(result);
  return grpc::Status::OK;
}

grpc::Status MetaServiceImpl::Mkfs(grpc::ServerContext *context,
    const MkfsRequest *request, MkfsReply *reply) {
  int result = backend_->Mkfs(request->fsid());
  reply->set_result(result);
  if (result == 0) {
    reply->set_error("");
  } else {
    reply->set_error(strerror(result));
  }
  return grpc::Status::OK;
}

} // namespace oiofs
