/*
 *
 * Copyright 2015, OpenIO
 *
 */


#include <glog/logging.h>

#include "src/page_list.h"

#include <list>

#ifndef FLOG
# ifdef __GNUC__
#  define FLOG(level) DVLOG(level) << __PRETTY_FUNCTION__
# else
#  define FLOG(level) DVLOG(level) << __FUNCTION__
# endif
#endif


void PageList::FreeList(fdpage_list_t& list) {
  for (auto iter = list.begin(); iter != list.end(); iter = list.erase(iter)) {
    delete (*iter);
  }
  list.clear();
}

PageList::PageList(size_t size, bool is_loaded) {
  Init(size, is_loaded);
}

PageList::~PageList() {
  Clear();
}

void PageList::Clear() {
  PageList::FreeList(pages);
}

bool PageList::Init(size_t size, bool is_loaded) {
  Clear();
  fdpage* page = new fdpage(0, size, is_loaded);
  pages.push_back(page);
  return true;
}

size_t PageList::Size() const {
  if (pages.empty()) {
    return 0;
  }
  auto riter = pages.rbegin();
  return static_cast<size_t>((*riter)->next());
}

bool PageList::Resize(size_t size, bool is_loaded) {
  size_t total = Size();

  if (0 == total) {
    Init(size, is_loaded);
  } else if (total < size) {
    // add new page
    fdpage *page = new fdpage(static_cast<off_t>(total), (size - total), is_loaded);
    FLOG(2) << " adding new " << (is_loaded?"":"un") << "loaded page ["
        << page->offset << ", " << page->end() << "]";
    pages.push_back(page);
  } else if (size < total) {
    // remove pages
    for (auto iter = pages.begin(); iter != pages.end();) {
      if (static_cast<size_t>((*iter)->next()) <= size) {
        ++iter;
      } else {
        if (size <= static_cast<size_t>((*iter)->offset)) {
          FLOG(2) << " removing page [" << (*iter)->offset << ", "
              << (*iter)->end() << "]";
          iter = pages.erase(iter);
        } else {
          (*iter)->bytes = size - static_cast<size_t>((*iter)->offset);
        }
      }
    }
  } else {
    // noop
  }
  return Compress();
}

bool PageList::IsPageLoaded(off_t start, size_t size) const {
  for (auto iter = pages.begin(); iter != pages.end(); ++iter) {
    if ((*iter)->end() < start) {
      continue;
    }
    if (!(*iter)->loaded) {
      return false;
    }
    if ((0 != size)
        && (static_cast<size_t>(start + size) <= static_cast<size_t>((*iter)->next()))) {
      break;
    }
  }
  return true;
}

bool PageList::SetPageLoadedStatus(off_t start, size_t size, bool is_loaded, bool is_compressed) {
  size_t current_size = Size();

  if (current_size <= static_cast<size_t>(start)) {
    if (current_size < static_cast<size_t>(start)) {
      // The new pages are further than the current end of file,
      // add missing pages between old end and new start.
      Resize(static_cast<size_t>(start), false);
    }
    // Add new pages at the end.
    Resize(static_cast<size_t>(start + size), is_loaded);
  }
  else if (current_size <= static_cast<size_t>(start + size)) {
    // The new pages overlap the end of the file,
    // unload pages between new start and old end by shrinking the file.
    Resize(static_cast<size_t>(start), false);
    // Add new pages at the end.
    Resize(static_cast<size_t>(start + size), is_loaded);
  }
  else {
    // Ensure there is a page starting precisely at `start`.
    Parse(start);
    // Ensure there is a page ending precisely at `start + size`.
    Parse(start + size);

    for (auto iter = pages.begin(); iter != pages.end(); ++iter) {
      if ((*iter)->end() < start) {
        // Iterator page ends before `start`.
        continue;
      } else if ((*iter)->offset >= static_cast<off_t>(start + size)) {
        // Iterator page starts after `start`.
        break;
      } else {
        // Iterator page overlaps [start, start+size]
        (*iter)->loaded = is_loaded;
        FLOG(2) << " page [" << (*iter)->offset << ", " << (*iter)->end()
            << "] is now " << (is_loaded?"loaded":"unloaded");
      }
    }
  }
  return (is_compressed ? Compress(): true);
}

bool PageList::FindUnloadedPage(off_t start, off_t& resstart, size_t& ressize) const {
  for (auto iter = pages.begin(); iter != pages.end(); ++iter) {
    if (start <= (*iter)->end()) {
      if (!(*iter)->loaded) {
        resstart = (*iter)->offset;
        ressize = (*iter)->bytes;
        return true;
      }
    }
  }
  return false;
}

size_t PageList::GetTotalUnloadedPageSize(off_t start, size_t size) const {
  if (0 == size) {
    if (static_cast<size_t>(start) < Size()) {
      size = static_cast<size_t>(Size() - start);
    }
  }
  size_t restsize = 0;
  off_t next = static_cast<off_t>(start + size);
  for (auto iter = pages.begin(); iter != pages.end(); ++iter) {
    if ((*iter)->next() <= start) {
      continue;
    }
    if (next <= (*iter)->offset) {
      break;
    }
    if ((*iter)->loaded) {
      continue;
    }
    size_t tmpsize;
    if ((*iter)->offset <= start) {
      if ((*iter)->next() <= next) {
        tmpsize = static_cast<size_t>((*iter)->next() - start);
      } else {
        tmpsize = static_cast<size_t>(next - start);
      }
    } else {
      if ((*iter)->next() <= next) {
        tmpsize = static_cast<size_t>((*iter)->next() - (*iter)->offset);
      } else {
        tmpsize = static_cast<size_t>(next - (*iter)->offset);
      }
    }
    restsize += tmpsize;
  }
  return restsize;
}

int PageList::GetUnloadedPages(fdpage_list_t& unloaded_list, off_t start, size_t size) const {
  if (0 == size) {
    if (static_cast<size_t>(start) < Size()) {
      size = static_cast<size_t>(Size() - start);
    }
  }
  // Offset of page immediately following the provided range
  off_t next = static_cast<off_t>(start + size);

  for (auto iter = pages.begin(); iter != pages.end(); ++iter) {
    if ((*iter)->next() <= start) {
      // Iterator page ends before `start`.
      continue;
    }
    if (next <= (*iter)->offset) {
      // Iterator page starts after `start`.
      break;
    }
    if ((*iter)->loaded) {
      continue;
    }

    off_t  page_start = std::max((*iter)->offset, start);
    off_t  page_next  = std::min((*iter)->next(), next);
    size_t page_size  = static_cast<size_t>(page_next - page_start);

    auto riter = unloaded_list.rbegin();
    if (riter != unloaded_list.rend() && (*riter)->next() == page_start) {
      (*riter)->bytes += page_size;
    } else {
      struct fdpage* page = new fdpage(page_start, page_size, false);
      unloaded_list.push_back(page);
    }
  }
  return unloaded_list.size();
}

bool PageList::Compress() {
  bool is_first = true;
  bool is_last_loaded = false;
  for (auto iter = pages.begin(); iter != pages.end(); ) {
    if (is_first) {
      is_first = false;
      is_last_loaded = (*iter)->loaded;
      ++iter;
    } else {
      if (is_last_loaded == (*iter)->loaded) {
        fdpage_list_t::iterator biter = iter;
        --biter;
        FLOG(2) << " merging " << (is_last_loaded? "":"un")
            << "loaded pages [" << (*biter)->offset << ", "
            << (*biter)->end() << "] and [" << (*iter)->offset << ", "
            << (*iter)->end() << "]";
        (*biter)->bytes += (*iter)->bytes;
        iter = pages.erase(iter);
      } else {
        is_last_loaded = (*iter)->loaded;
        ++iter;
      }
    }
  }
  return true;
}

bool PageList::Parse(off_t new_pos) {
  for (auto iter = pages.begin(); iter != pages.end(); ++iter) {
    if (new_pos == (*iter)->offset) {
      return true;
    } else if (((*iter)->offset < new_pos) && (new_pos < (*iter)->next())) {
      // Create a new page just before new_pos
      struct fdpage *page = new fdpage((*iter)->offset,
          static_cast<size_t>(new_pos - (*iter)->offset), (*iter)->loaded);
      FLOG(2) << " Inserting new " << (page->loaded? "":"un")
          << "loaded page [" << page->offset << ", " << page->end() << "]";
      pages.insert(iter, page);

      FLOG(2) << " Shrinking " << ((*iter)->loaded? "":"un")
          << "loaded page [" << (*iter)->offset << ", "
          << (*iter)->end() << "] to [" << new_pos << ", " << (*iter)->end()
          << "]";
      // Shrink the current page so its start matches new_pos
      (*iter)->bytes -= (new_pos - (*iter)->offset);
      (*iter)->offset = new_pos;

      return true;
    }
  }
  return false;
}


