/*
 *
 * Copyright 2015, OpenIO
 *
 */

#include <errno.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <dirent.h>
#include <gtest/gtest.h>
#include <array>
#include <string>
#include <iostream>
#include "src/meta/client.h"


TEST(Client, InodeRef) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());

  Inode *in;
  auto r = client.GetInode(OIOFS_INO_ROOT, &in);
  ASSERT_EQ(r, 0);
  ASSERT_TRUE(nullptr != in);
  client.PutInode(in);
}

TEST(Client, ReadWrite) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *root_in = nullptr;
  client.GetInode(OIOFS_INO_ROOT, &root_in);
  ASSERT_TRUE(nullptr != root_in);

  Inode *out = nullptr;
  int a_uid = -1;
  int a_gid = -1;

  FileHandle *fh = nullptr;
  struct stat attr;
  char c_file[256];
  snprintf(c_file, sizeof(c_file), "test_readwrite_file_%d", getpid());
  ASSERT_EQ(0, client.Create(root_in, c_file, 0666, O_CREAT, &attr, a_uid, a_gid, &out, &fh));
  ASSERT_TRUE(nullptr != fh);
  char out0[] = "もち";
  char out1[] = "world\n";

  char in0[sizeof(out0)];
  char in1[sizeof(out1)];

  ASSERT_EQ(sizeof(out0), client.Write(fh, out0, 0, sizeof(out0)));
  ASSERT_EQ(sizeof(out0), client.Read(fh, 0, sizeof(out0), in0));
  ASSERT_EQ(0, client.Flush(fh));
	ASSERT_EQ(0, client.Release(fh));
}

TEST(Client, Readdir) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *in = nullptr;
  client.GetInode(1, &in);
  ASSERT_TRUE(nullptr != in);

  Inode *dir = nullptr;
  Inode *out = nullptr;
  int a_uid = -1;
  int a_gid = -1;
  struct stat attr = {0};
  struct oiofs::dir_result_t *dirp = nullptr;

  char c_dir[256];
  snprintf(c_dir, sizeof(c_dir), "test_readdir_%d", getpid());
  ASSERT_EQ(0, client.Mkdir(in, c_dir, 0755, &attr, a_uid, a_gid, &dir));
  ASSERT_EQ(0, client.Opendir(dir, a_uid, a_gid, &dirp));

  int buflen = 1024;
	int nread;
	struct dirent *d;
  char *buf = new char[buflen];

  // dir is empty only . and .. entries
  nread = client.Getdents(dirp, buf, buflen);
	ASSERT_GT(nread, 0);
	ASSERT_EQ(2 * sizeof(struct dirent), nread);

	d = (struct dirent*)(buf);
	ASSERT_STREQ(".", d->d_name);
	ASSERT_EQ(dir->ino, d->d_ino);
	d++;
	ASSERT_STREQ("..", d->d_name);
	ASSERT_EQ(in->ino, d->d_ino);

  FileHandle *fh = nullptr;
  char c_file[256];
  snprintf(c_file, sizeof(c_file), "foo");
  ASSERT_EQ(0, client.Create(dir, c_file, 0777, O_CREAT, &attr, a_uid, a_gid, &out, &fh));
  ASSERT_EQ(0, client.Flush(fh));
	ASSERT_EQ(0, client.Release(fh));

  // dir not empty
  ASSERT_EQ(0, client.Releasedir(dirp));
  ASSERT_EQ(0, client.Opendir(dir, a_uid, a_gid, &dirp));
  nread = client.Getdents(dirp, buf, buflen);

	ASSERT_GT(nread, 0);
	ASSERT_EQ(3 * sizeof(struct dirent), nread);
	d = (struct dirent*)(buf);
	ASSERT_STREQ(".", d->d_name);
	ASSERT_EQ(dir->ino, d->d_ino);
	d++;
	ASSERT_STREQ("..", d->d_name);
	ASSERT_EQ(in->ino, d->d_ino);
	d++;
	ASSERT_STREQ("foo", d->d_name);
	ASSERT_EQ(out->ino, d->d_ino);

  // small buffer
  ASSERT_EQ(0, client.Releasedir(dirp));
  ASSERT_EQ(0, client.Opendir(dir, a_uid, a_gid, &dirp));
  ASSERT_EQ(-ERANGE, client.Getdents(dirp, buf, 1));

  // paginate list
  ASSERT_EQ(0, client.Releasedir(dirp));
  ASSERT_EQ(0, client.Opendir(dir, a_uid, a_gid, &dirp));

	buflen = sizeof(struct dirent);
  nread = client.Getdents(dirp, buf, buflen);
	ASSERT_GT(nread, 0);
	ASSERT_EQ(buflen, nread);
	d = (struct dirent*)(buf);
	ASSERT_STREQ(".", d->d_name);
	ASSERT_EQ(dir->ino, d->d_ino);

	buflen = 2 * sizeof(struct dirent);
  nread = client.Getdents(dirp, buf, buflen);
	ASSERT_GT(nread, 0);
	ASSERT_EQ(buflen, nread);
	d = (struct dirent*)(buf);
	ASSERT_STREQ("..", d->d_name);
	ASSERT_EQ(in->ino, d->d_ino);
	d++;
	ASSERT_STREQ("foo", d->d_name);
	ASSERT_EQ(out->ino, d->d_ino);

  // TODO cleanup
  ASSERT_EQ(0, client.Releasedir(dirp));
}

TEST(Client, ReaddirAdvanced) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *in = nullptr;
  client.GetInode(1, &in);
  ASSERT_TRUE(nullptr != in);

  Inode *dir = nullptr;
  Inode *out = nullptr;
  int a_uid = -1;
  int a_gid = -1;
  struct stat attr = {0};
  struct oiofs::dir_result_t *dirp = nullptr;

  char c_dir[256];
  snprintf(c_dir, sizeof(c_dir), "test_readdir_advanced_%d", getpid());
  ASSERT_EQ(0, client.Mkdir(in, c_dir, 0755, &attr, a_uid, a_gid, &dir));
  ASSERT_EQ(0, client.Opendir(dir, a_uid, a_gid, &dirp));

	FileHandle *fh;
	char c_file[256];
	int i = 0, r = 64;
	for (; i < r; ++i) {
		snprintf(c_file, sizeof(c_file), "test_file_%d", i);
		ASSERT_EQ(0, client.Create(dir, c_file, 0777, O_CREAT, &attr, a_uid, a_gid, &out, &fh));
		ASSERT_EQ(0, client.Flush(fh));
		ASSERT_EQ(0, client.Release(fh));
	}

  ASSERT_EQ(0, client.Opendir(dir, a_uid, a_gid, &dirp));

  struct dirent *result;
  result = client.Readdir(dirp);
  ASSERT_TRUE(nullptr != result);
  ASSERT_STREQ(".", result->d_name);
  ASSERT_EQ(dir->ino, result->d_ino);

  result = client.Readdir(dirp);
  ASSERT_TRUE(nullptr != result);
  ASSERT_STREQ("..", result->d_name);
  ASSERT_EQ(in->ino, result->d_ino);

  std::vector<std::pair<char *, int>> entries;
  for (i = 0; i < r; ++i) {
    result = client.Readdir(dirp);
    ASSERT_TRUE(result != nullptr);
    entries.push_back(std::pair<char *, int>(strdup(result->d_name), 0));
  }

  ASSERT_TRUE(nullptr == client.Readdir(dirp));

  // test rewinddir
  client.Rewinddir(dirp);

  result = client.Readdir(dirp);
  ASSERT_TRUE(nullptr != result);
  ASSERT_STREQ(".", result->d_name);
  ASSERT_EQ(dir->ino, result->d_ino);

  result = client.Readdir(dirp);
  ASSERT_TRUE(nullptr != result);
  ASSERT_STREQ("..", result->d_name);
  ASSERT_EQ(in->ino, result->d_ino);

  for (i = 0; i < r - 1; ++i) {
    int off = client.Telldir(dirp);
    ASSERT_GT(off, -1);
    client.Seekdir(dirp, off);
    result = client.Readdir(dirp);
    ASSERT_TRUE(nullptr != result);
    ASSERT_STREQ(entries[i].first, result->d_name);
  }

  client.Rewinddir(dirp);

  int t = client.Telldir(dirp);
  ASSERT_GT(t, -1);

  ASSERT_TRUE(nullptr != client.Readdir(dirp));

  // test seekdir
  client.Seekdir(dirp, t);

  // test getdents
  struct dirent *dentries;
  dentries = (struct dirent *)malloc(r * sizeof(struct dirent));

  int count = 0;
  while (count < r) {
    int len = client.Getdents(dirp, (char *)dentries, r * sizeof(struct dirent));
    ASSERT_GT(len, 0);
    ASSERT_TRUE(0 == (len % sizeof(struct dirent)));
    int n = len / sizeof(struct dirent);
    if (0 == count) {
      ASSERT_STREQ(".", dentries[0].d_name);
      ASSERT_STREQ("..", dentries[1].d_name);
    }
    int j;
    i = count;
    for (j = 2; j < n; ++i, ++j) {
      ASSERT_STREQ(entries[i].first, dentries[j].d_name);
    }
    count += n;
  }

  free(dentries);

  ASSERT_EQ(0, client.Releasedir(dirp));
}

// no access check support for now
TEST(Client, DISABLED_Access) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *in = nullptr;
  client.GetInode(1, &in);
  ASSERT_TRUE(nullptr != in);

  Inode *dir = nullptr;
  Inode *out = nullptr;
  int a_uid = -1;
  int a_gid = -1;

  int u_uid = -1;
  int u_gid = -1;
  int mask = 0;
  struct stat attr;
  auto test_path = "test";
  ASSERT_EQ(client.Mkdir(in, test_path, 0755, &attr, a_uid, a_gid, &dir), 0);

  // user
  // chmod 0700
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  // chown 123 456
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "u1", 0755, &attr, u_uid, u_gid, &out), 0);
  // chown 1 456
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);

  // group
  // chmod 0770
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  // chown 1 456
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "u2", 0755, &attr, u_uid, u_gid, &out), 0);
  // chown 1 2
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);

  // user overrides group
  // chmod 0470
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  // chown 123 456
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);

  // other
  // chmod 0777
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  // chown 1 1
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "u3", 0755, &attr, u_uid, u_gid, &out), 0);
  // chmod 0770
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);

  // user and group overrides other
  // chmod 07
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  // chown 1 456
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);
  // chown 123 1
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);
  // chown 123 456
  ASSERT_EQ(client.Setattr(dir, &attr, mask, a_uid, a_gid), 0);
  ASSERT_EQ(client.Mkdir(dir, "no", 0755, &attr, u_uid, u_gid, &out), -EACCES);

  // cleanup
  ASSERT_EQ(client.Rmdir(dir, "u1", a_uid, a_gid), 0);
  ASSERT_EQ(client.Rmdir(dir, "u2", a_uid, a_gid), 0);
  ASSERT_EQ(client.Rmdir(dir, "u3", a_uid, a_gid), 0);
  ASSERT_EQ(client.Rmdir(in, test_path, a_uid, a_gid), 0);

  // shutdown?
}

// no access path check support for now
TEST(Client, DISABLED_AccessPath) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *in = nullptr;
  client.GetInode(1, &in);
  ASSERT_TRUE(nullptr != in);

  Inode *good = nullptr;
  Inode *bad = nullptr;
  Inode *out = nullptr;

  FileHandle *fh = nullptr;
  struct stat attr;
  auto good_path = "good";
  auto bad_path = "bad";

  int flags = 0;
  int a_uid = -1;
  int a_gid = -1;

  int u_uid = -1;
  int u_gid = -1;

  ASSERT_EQ(client.Mkdir(in, good_path, 0755, &attr, a_uid, a_gid, &good), 0);
  ASSERT_EQ(client.Mkdir(good, "p", 0755, &attr, a_uid, a_gid, &out), 0);
  ASSERT_EQ(client.Mkdir(in, bad_path, 0755, &attr, a_uid, a_gid, &bad), 0);
  ASSERT_EQ(client.Mkdir(bad, "p", 0755, &attr, a_uid, a_gid, &out), 0);

  client.Create(good, "q", 0755, flags, &attr, a_uid, a_gid, &out, &fh);
  client.Release(fh);
  client.Create(bad, "q", 0755, flags, &attr, a_uid, a_gid, &out, &fh);
  client.Release(fh);
  client.Create(bad, "z", 0755, flags, &attr, a_uid, a_gid, &out, &fh);
  client.Write(fh, "TEST FAILURE", 0, 12); // admin
  client.Release(fh);

  // allowed ops
  ASSERT_GE(client.Mkdir(good, "x", 0755, &attr, u_uid, u_gid, &out), 0);
  ASSERT_GE(client.Rmdir(good, "p", u_uid, u_gid), 0);
  ASSERT_GE(client.Unlink(good, "q", u_uid, u_gid), 0);

  client.Create(good, "y", 0755, flags, &attr, u_uid, u_gid, &out, &fh);
  client.Write(fh, "test", 0, 4);
  client.Release(fh);

  ASSERT_GE(client.Unlink(good, "y", u_uid, u_gid), 0);
  ASSERT_GE(client.Rmdir(good, "x", u_uid, u_gid), 0);

  client.Create(bad, "z", 0644, flags, &attr, u_uid, u_gid, &out, &fh);
  client.Release(fh);

  // not allowed ops
  ASSERT_GE(client.Mkdir(bad, "x", 0755, &attr, u_uid, u_gid, &out), 0);
  ASSERT_GE(client.Rmdir(bad, "p", u_uid, u_gid), 0);
  ASSERT_GE(client.Unlink(bad, "q", u_uid, u_gid), 0);

  auto r = client.Create(bad, "y", 0755, flags, &attr, u_uid, u_gid, &out, &fh);
  ASSERT_LT(r, 0);

  // cleanup
  ASSERT_EQ(client.Unlink(bad, "p", a_uid, a_gid), 0);
  ASSERT_EQ(client.Unlink(bad, "z", a_uid, a_gid), 0);
  ASSERT_EQ(client.Rmdir(bad, "q", a_uid, a_gid), 0);
  ASSERT_EQ(client.Unlink(bad, "asdf", a_uid, a_gid), 0);
  ASSERT_EQ(client.Rmdir(in, good_path, a_uid, a_gid), 0);
  ASSERT_EQ(client.Rmdir(in, bad_path, a_uid, a_gid), 0);

  // shutdown
}

TEST(Client, ReadEmpty) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *root_in = nullptr;
  client.GetInode(1, &root_in);
  ASSERT_TRUE(nullptr != root_in);

  Inode *out = nullptr;
  int a_uid = -1;
  int a_gid = -1;

  char c_file[256];
  struct stat attr;
  FileHandle *fh = nullptr;
  snprintf(c_file, sizeof(c_file), "test_readempty_%d", getpid());
  ASSERT_EQ(0, client.Create(root_in, c_file, 0777, O_CREAT, &attr, a_uid, a_gid, &out, &fh));
  ASSERT_EQ(0, client.Flush(fh));
	ASSERT_EQ(0, client.Release(fh));

  ASSERT_EQ(0, client.Open(out, O_RDONLY, 0, a_uid, a_gid, &fh));


  char buf[4096];
  ASSERT_EQ(0, client.Read(fh, 0, 4096, buf));

  // shutdown
}

TEST(Client, Rename) {
  oiofs::Client client;

  ASSERT_EQ(0, client.Mount());
  Inode *root_in = nullptr;
  client.GetInode(1, &root_in);
  ASSERT_TRUE(nullptr != root_in);

  Inode *out = nullptr;
  int a_uid = -1;
  int a_gid = -1;

  char src_file[256];
  char dst_file[256];

  struct stat attr;
  FileHandle *fh = nullptr;
  snprintf(src_file, sizeof(src_file), "test_rename_src_%d", getpid());
  ASSERT_EQ(0, client.Create(root_in, src_file, 0777, O_CREAT, &attr, a_uid, a_gid, &out, &fh));
  ASSERT_EQ(0, client.Flush(fh));
	ASSERT_EQ(0, client.Release(fh));

  snprintf(dst_file, sizeof(dst_file), "test_rename_dst_%d", getpid());
  ASSERT_EQ(0, client.Rename(root_in, src_file, root_in, dst_file, a_uid, a_gid));

  // verify dst file
  ASSERT_EQ(0, client.Lookup(root_in, dst_file, &attr, a_uid, a_gid, &out));

  // verify src file
  ASSERT_EQ(-ENOENT, client.Lookup(root_in, src_file, &attr, a_uid, a_gid, &out));

  // rename again
  ASSERT_EQ(-ENOENT, client.Rename(root_in, src_file, root_in, dst_file, a_uid, a_gid));

  ASSERT_EQ(0, client.Unlink(root_in, dst_file, a_uid, a_gid));
  // shutdown
}
