/*
 *
 * Copyright 2015, OpenIO
 *
 */

#include <errno.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <dirent.h>
#include <oiofs/oiofs.h>
#include <gtest/gtest.h>
#include <string>


TEST(OioFS, UseUnmounted) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));

  struct statvfs stv;
  EXPECT_EQ(-ENOTCONN, oiofs_statfs(minfo, "/", &stv));
  EXPECT_EQ(-ENOTCONN, oiofs_link(minfo, "/", "/foo"));
  EXPECT_EQ(-ENOTCONN, oiofs_unlink(minfo, "/foo"));
  EXPECT_EQ(-ENOTCONN, oiofs_rename(minfo, "/foo", "/bar"));
  EXPECT_EQ(-ENOTCONN, oiofs_mkdir(minfo, "/foo", 0655));
  EXPECT_EQ(-ENOTCONN, oiofs_rmdir(minfo, "/foo"));
  EXPECT_EQ(-ENOTCONN, oiofs_readlink(minfo, "/foo", nullptr, 0));

  struct stat st;
  EXPECT_EQ(-ENOTCONN, oiofs_stat(minfo, "/foo", &st));
  EXPECT_EQ(-ENOTCONN, oiofs_setattr(minfo, "/foo", &st, 0));
  EXPECT_EQ(-ENOTCONN, oiofs_chmod(minfo, "/foo", 0));
  EXPECT_EQ(-ENOTCONN, oiofs_chown(minfo, "/foo", 0, 0));

  // TODO(noone) add others calls
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, MountRemount) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));
  ASSERT_EQ(0, oiofs_unmount(minfo));

  ASSERT_EQ(0, oiofs_mount(minfo));
  ASSERT_EQ(0, oiofs_unmount(minfo));

  ASSERT_EQ(0, oiofs_mount(minfo));

  ASSERT_EQ(0, oiofs_unmount(minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, UnmountUnmounted) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(-ENOTCONN, oiofs_unmount(minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, ReleaseUnmounted) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, ReleaseMounted) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));
  ASSERT_EQ(-EISCONN, oiofs_release(minfo));
  ASSERT_EQ(0, oiofs_unmount(minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, UnmountRelease) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));
  ASSERT_EQ(0, oiofs_unmount(minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, Symlink) {
  struct oiofs_mount_info* minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));

  char test_file[256];

  int fd = oiofs_open(minfo, test_file, O_CREAT|O_RDWR, 0666);
  ASSERT_GT(fd, 0);
  ASSERT_EQ(0, oiofs_close(minfo, fd));

  char test_symlink[256];

  ASSERT_EQ(0, oiofs_symlink(minfo, test_file, test_symlink));

  // NOFOLLOW
  fd = oiofs_open(minfo, test_symlink, O_NOFOLLOW, 0);
  ASSERT_EQ(fd, -ELOOP);

  struct stat st;
  ASSERT_EQ(0, oiofs_stat(minfo, test_file, &st));

  struct stat stl;
  ASSERT_EQ(0, oiofs_stat(minfo, test_symlink, &stl));

  ASSERT_TRUE(!memcmp(&st, &stl, sizeof(st)));

  fd = oiofs_open(minfo, test_file, O_CREAT|O_RDWR, 0666);
  ASSERT_GT(fd, 0);

  oiofs_close(minfo, fd);

  ASSERT_EQ(0, oiofs_symlink(minfo, test_file, test_symlink));

  ASSERT_EQ(0, oiofs_stat(minfo, test_file, &st));

  ASSERT_EQ(0, oiofs_stat(minfo, test_symlink, &stl));

  ASSERT_TRUE(!memcmp(&st, &stl, sizeof(st)));

  ASSERT_EQ(0, oiofs_unmount(minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, ReadEmptyFile) {
  struct oiofs_mount_info *minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));

  char path[256];
  int fd = oiofs_open(minfo, path, O_CREAT|O_TRUNC|O_WRONLY, 0644);
  ASSERT_GT(fd, 0);
  ASSERT_EQ(0, oiofs_close(minfo, fd));

  fd = oiofs_open(minfo, path, O_RDONLY, 0);
  ASSERT_GT(fd, 0);

  char buf[4096];
  ASSERT_EQ(0, oiofs_read(minfo, fd, buf, O_RDONLY, 0));
  ASSERT_EQ(0, oiofs_close(minfo, fd));

  ASSERT_EQ(0, oiofs_unmount(minfo));
  ASSERT_EQ(0, oiofs_release(minfo));
}

TEST(OioFS, Readdir) {
  struct oiofs_mount_info *minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));

  struct oiofs_dir_result* dir_result;
  char path[256];
  ASSERT_EQ(-ENOENT, oiofs_opendir(minfo, path, &dir_result));

  ASSERT_EQ(0, oiofs_mkdir(minfo, path, 0777));
  struct stat stat_buf;
  ASSERT_EQ(0, oiofs_stat(minfo, path, &stat_buf));
  ASSERT_NE(0, S_ISDIR(stat_buf.st_mode));

  char path2[256];
  ASSERT_EQ(-ENOENT, oiofs_stat(minfo, path2, &stat_buf));

  char path3[256];
  int i = 0, r = 10;

  for (; i < r; ++i) {
    int fd = oiofs_open(minfo, path3, O_CREAT|O_RDONLY, 0666);
    ASSERT_GT(fd, 0);
    ASSERT_EQ(0, oiofs_close(minfo, fd));
  }

  ASSERT_EQ(0, oiofs_opendir(minfo, path2, &dir_result));

  // TODO(noone)
}

TEST(OioFS, Rename) {
  struct oiofs_mount_info *minfo;
  ASSERT_EQ(0, oiofs_create(&minfo));
  ASSERT_EQ(0, oiofs_mount(minfo));

  char path_src[256];
  char path_dst[256];

  int fd = oiofs_open(minfo, path_src, O_CREAT|O_TRUNC|O_WRONLY, 0777);

  ASSERT_GT(fd, 0);
  ASSERT_EQ(0, oiofs_close(minfo, fd));

  ASSERT_EQ(0, oiofs_rename(minfo, path_src, path_dst));

  struct stat st;
  ASSERT_EQ(0, oiofs_stat(minfo, path_dst, &st));

  ASSERT_EQ(-ENOENT, oiofs_stat(minfo, path_src, &st));

  ASSERT_EQ(-ENOENT, oiofs_rename(minfo, path_src, path_dst));

  ASSERT_EQ(0, oiofs_unlink(minfo, path_dst));

  ASSERT_EQ(0, oiofs_unmount(minfo));

  ASSERT_EQ(0, oiofs_release(minfo));
}
