/*
OpenIO SDS sqliterepo
Copyright (C) 2014 Worldine, original work as part of Redcurrant
Copyright (C) 2015 OpenIO, modified as part of OpenIO Software Defined Storage

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 3.0 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.
*/

#include <stddef.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>

#include <metautils/lib/metautils.h>

#include "sqliterepo.h"
#include "hash.h"
#include "election.h"
#include "version.h"
#include "sqlx_remote.h"
#include "synchro.h"
#include "gridd_client_pool.h"
#include "internals.h"

#define EVENTLOG_SIZE 16
#define STATUS_FINAL(e) ((e) >= STEP_LOST)

typedef guint req_id_t;

enum sqlx_action_e
{
	ACTION_NONE,
	ACTION_PING,
	ACTION_FAIL,
	ACTION_RETRY, /* re-send getvers */
	ACTION_RESTART,
	ACTION_LEAVE,
	ACTION_EXPIRE
};

enum election_step_e
{
	STEP_NONE = 0,
	STEP_CANDREQ,
	STEP_CANDOK,
	STEP_LEAVING,
	STEP_PRELOST,
	STEP_PRELEAD,
	STEP_LOST,
	STEP_LEADER,
	STEP_FAILED,
#define STEP_MAX (STEP_FAILED+1)
};

enum event_type_e
{
	EVT_NONE = 0,
	EVT_DISCONNECTED,
	EVT_EXITING,

	EVT_GETVERS_OK,
	EVT_GETVERS_OUTDATED,
	EVT_GETVERS_CONCURRENT,
	EVT_GETVERS_ERROR,

	EVT_CREATE_OK,
	EVT_CREATE_KO,

	EVT_NODE_LEFT,

	EVT_MASTER_KO,
	EVT_MASTER_EMPTY,
	EVT_MASTER_OK,
	EVT_MASTER_CHANGE,

	EVT_RESYNC_REQ,
	EVT_RESYNC_DONE,

	EVT_LIST_OK,
	EVT_LIST_KO,

	EVT_LEAVE_OK,
	EVT_LEAVE_KO,
};

/* @private */
struct logged_event_s
{
	enum event_type_e event   :8;
	enum election_step_e pre  :8;
	enum election_step_e post :8;
};

/* @private */
struct deque_beacon_s
{
	guint count;
	struct election_member_s *front;
	struct election_member_s *back;
};

struct election_manager_s
{
	struct election_manager_vtable_s *vtable;

	struct sqlx_peering_s *peering;
	struct sqlx_sync_s *sync;

	/* GTree<hashstr_t*,GCond*> */
	GTree *conditions;

	GThreadPool *completions;

	/* do not free or change the fields below */
	const struct replication_config_s *config;

	GMutex lock;
	GTree *members_by_key;
	struct deque_beacon_s members_by_state[STEP_MAX];

	/* how long we accept to wait for a status. */
	gint64 delay_wait;

	/* how long we wait before expiring a base */
	gint64 delay_expire_none;
	gint64 delay_expire_final;
	gint64 delay_expire_final_leader;
	gint64 delay_expire_failed;

	/* how long we wait before failing a stalled election */
	gint64 delay_fail_pending;

	/* how long we wait before restarting a failed election. */
	gint64 delay_retry_pending;
	gint64 delay_retry_failed;

	/* how long we wait after the last USE sent to send a new */
	gint64 delay_ping_pending;
	gint64 delay_ping_final;
	gint64 delay_ping_failed;

	/* When has the lock been acquired */
	gint64 when_lock;

	gboolean exiting;
	gboolean synchronous_completions;

	req_id_t next_id;
};

struct election_member_s
{
	struct election_member_s *prev;
	struct election_member_s *next;

	struct election_manager_s *manager;

	/* Weak pointer to the condition, do not free! */
	GCond *cond;

	struct sqlx_name_mutable_s name;
	hashstr_t *key;

	gint64 last_status; /* last time the status was changed */
	gint64 last_USE; /* last time a USE request has been sent */
	gint64 last_atime; /* last time the app wanted a status */

	gint64 myid; /* ID generated by zookeeper */
	gint64 master_id; /* ID of the master */
	gchar *master_url; /* First node of the children sequence (sorted by ID) */

	guint refcount;

	req_id_t reqid_PIPEFROM;
	req_id_t reqid_GETVERS;

	guint16 concurrent_GETVERS;
	guint16 errors_GETVERS;
	guint16 pending_GETVERS;

	guint log_index : 8;
	enum election_step_e step : 8;

	unsigned char requested_USE : 1;
	unsigned char requested_PIPEFROM : 1;
	unsigned char requested_EXIT : 1;

	unsigned char delayed_NODE_LEFT : 1;
	unsigned char delayed_MASTER_CHANGE : 1;

	/* PIPEFROM being requested */
	unsigned char pending_PIPEFROM : 1;
	/* Monitor already set in Zookeeper */
	unsigned char pending_watch : 1;

	struct logged_event_s log[EVENTLOG_SIZE];
};

gint64 oio_election_period_cond_wait = G_TIME_SPAN_SECOND;

static void _noop (gpointer p) { (void)p; }

static GPrivate th_local_key_manager = G_PRIVATE_INIT(_noop);

/* ------------------------------------------------------------------------- */

static void member_unref(struct election_member_s *member);
static void member_destroy(struct election_member_s *member);

static void _manager_clean(struct election_manager_s *manager);

enum election_mode_e _manager_get_mode (const struct election_manager_s *manager);

const char * _manager_get_local (const struct election_manager_s *manager);

static GError * _election_get_peers(struct election_manager_s *manager,
		const struct sqlx_name_s *n, gboolean nocache, gchar ***peers);

static GError * _election_trigger_RESYNC(struct election_manager_s *manager,
		const struct sqlx_name_s *n);

static GError * _election_init(struct election_manager_s *manager,
		const struct sqlx_name_s *n);

static GError * _election_start(struct election_manager_s *manager,
		const struct sqlx_name_s *n);

static GError * _election_exit(struct election_manager_s *manager,
		const struct sqlx_name_s *n);

static enum election_status_e _election_get_status(struct election_manager_s *m,
		const struct sqlx_name_s *n, gchar **master_url);

static struct election_manager_vtable_s VTABLE =
{
	_manager_clean,
	_manager_get_mode,
	_manager_get_local,
	_election_get_peers,
	_election_init,
	_election_start,
	_election_exit,
	_election_get_status,
	_election_trigger_RESYNC,
};

static void transition_error(struct election_member_s *member,
		enum event_type_e evt, enum ZOO_ERRORS zrc);

static void transition(struct election_member_s *member,
		enum event_type_e evt_type, void *evt_arg);

static void defer_PIPEFROM(struct election_member_s *member);

static gboolean defer_USE(struct election_member_s *member);

static void defer_GETVERS(struct election_member_s *member);

static void member_debug(const char *func, const char *tag,
		const struct election_member_s *m);

static enum sqlx_action_e _member_get_next_action (
		const struct election_member_s *m);

static gboolean wait_for_final_status(struct election_member_s *m,
		gint64 deadline);

static int gint64_cmp(register gint64 i1, register gint64 i2) { return CMP(i1,i2); }

static int
gint64_sort(gconstpointer p1, gconstpointer p2)
{
	return gint64_cmp(*(gint64*)p1, *(gint64*)p2);
}

static void
_thlocal_set_manager (struct election_manager_s *manager)
{
	g_private_replace (&th_local_key_manager, manager);
}

#define _thlocal_get_manager() g_private_get (&th_local_key_manager)

static void
_cond_clean (gpointer p)
{
	GCond *cond = p;
	if (cond) {
		g_cond_clear (cond);
		g_free (cond);
	}
}

#define _manager_lock(M) do { \
	g_mutex_lock(&(M)->lock); \
	(M)->when_lock = g_get_monotonic_time(); \
} while (0)

#define _manager_unlock(M) do { \
	/* JFS: One might check here we did not spent to much time in the current \
	 * critical section */ \
	(M)->when_lock = 0; \
	g_mutex_unlock(&(M)->lock); \
} while (0)

static void _completion_router(gpointer p, gpointer u);

/* -------------------------------------------------------------------------- */

static void
_DEQUE_remove (struct election_member_s *m)
{
	EXTRA_ASSERT(m != NULL);
	EXTRA_ASSERT(m->step < STEP_MAX);
	struct deque_beacon_s *beacon = m->manager->members_by_state + m->step;
	EXTRA_ASSERT(beacon->count > 0);

	struct election_member_s *prev = m->prev, *next = m->next;
	if (beacon->front == m) beacon->front = next;
	if (beacon->back == m) beacon->back = prev;
	if (prev) prev->next = next;
	if (next) next->prev = prev;
	m->prev = m->next = NULL;
	-- beacon->count;
}

static void
_DEQUE_add (struct election_member_s *m)
{
	EXTRA_ASSERT(m != NULL);
	EXTRA_ASSERT(m->step < STEP_MAX);
	EXTRA_ASSERT(m->prev == NULL);
	EXTRA_ASSERT(m->next == NULL);
	struct deque_beacon_s *beacon = m->manager->members_by_state + m->step;

	if (beacon->back) {
		m->prev = beacon->back;
		beacon->back->next = m;
	}
	beacon->back = m;
	if (!beacon->front)
		beacon->front = m;
	++ beacon->count;
}

/* --- Misc helpers --------------------------------------------------------- */

static gboolean
_is_over (const gint64 last, const gint64 delay)
{
	const gint64 now = oio_ext_monotonic_time ();
	return (0 != delay && 0 != last && last < OLDEST(now,delay));
}

static GArray *
nodev_to_int64v(const struct String_vector *sv, const char *prefix)
{
	GArray *array = g_array_new(0, 0, sizeof(gint64));

	for (int32_t i = 0; i < sv->count; i++) {
		const char *s = sv->data[i];
		if (g_str_has_prefix(s, prefix)) {
			const char *stripe = strrchr(s, '-');
			if (stripe != NULL) {
				stripe ++;
				if (strlen(stripe) != 10)
					continue;
				gint64 i64 = 0;
				if (oio_str_is_number(stripe, &i64))
					g_array_append_vals(array, &i64, 1);
			}
		}
	}

	if (array->len > 1)
		g_array_sort(array, gint64_sort);

	return array;
}

static const char *
_action2str(enum sqlx_action_e action)
{
	switch (action) {
		ON_ENUM(ACTION_,NONE);
		ON_ENUM(ACTION_,PING);
		ON_ENUM(ACTION_,FAIL);
		ON_ENUM(ACTION_,RETRY);
		ON_ENUM(ACTION_,RESTART);
		ON_ENUM(ACTION_,LEAVE);
		ON_ENUM(ACTION_,EXPIRE);
	}

	g_assert_not_reached ();
	return "!INVALID!";
}

static const char *
_step2str(enum election_step_e step)
{
	switch (step) {
		ON_ENUM(STEP_,NONE);
		ON_ENUM(STEP_,CANDREQ);
		ON_ENUM(STEP_,CANDOK);
		ON_ENUM(STEP_,LEAVING);
		ON_ENUM(STEP_,PRELOST);
		ON_ENUM(STEP_,PRELEAD);
		ON_ENUM(STEP_,LOST);
		ON_ENUM(STEP_,LEADER);
		ON_ENUM(STEP_,FAILED);
	}

	g_assert_not_reached ();
	return "!INVALID!";
}

static const char *
_evt2str(enum event_type_e evt)
{
	switch (evt) {
		ON_ENUM(EVT_,NONE);
		ON_ENUM(EVT_,DISCONNECTED);
		ON_ENUM(EVT_,EXITING);
		ON_ENUM(EVT_,GETVERS_OK);
		ON_ENUM(EVT_,GETVERS_OUTDATED);
		ON_ENUM(EVT_,GETVERS_CONCURRENT);
		ON_ENUM(EVT_,GETVERS_ERROR);
		ON_ENUM(EVT_,CREATE_OK);
		ON_ENUM(EVT_,CREATE_KO);
		ON_ENUM(EVT_,NODE_LEFT);
		ON_ENUM(EVT_,MASTER_KO);
		ON_ENUM(EVT_,MASTER_EMPTY);
		ON_ENUM(EVT_,MASTER_OK);
		ON_ENUM(EVT_,MASTER_CHANGE);
		ON_ENUM(EVT_,RESYNC_REQ);
		ON_ENUM(EVT_,RESYNC_DONE);
		ON_ENUM(EVT_,LIST_OK);
		ON_ENUM(EVT_,LIST_KO);
		ON_ENUM(EVT_,LEAVE_OK);
		ON_ENUM(EVT_,LEAVE_KO);
	}

	g_assert_not_reached ();
	return "!INVALID!";
}

/* Public API --------------------------------------------------------------- */

static void
election_manager_dump_delays(struct election_manager_s *manager)
{
	GRID_INFO("Election delays:");
	GRID_INFO("- get_status=%"G_GINT64_FORMAT"ms",
			manager->delay_wait / G_TIME_SPAN_MILLISECOND);
	GRID_INFO("- expire_final=%"G_GINT64_FORMAT"ms, expire_final_leader=%"G_GINT64_FORMAT"ms",
			manager->delay_expire_final / G_TIME_SPAN_MILLISECOND,
			manager->delay_expire_final_leader / G_TIME_SPAN_MILLISECOND);
	GRID_INFO("- expire_none=%"G_GINT64_FORMAT"ms, expire_failed=%"G_GINT64_FORMAT"ms",
			manager->delay_expire_none / G_TIME_SPAN_MILLISECOND,
			manager->delay_expire_failed / G_TIME_SPAN_MILLISECOND);
	GRID_INFO("- retry_pending=%"G_GINT64_FORMAT"ms, retry_failed=%"G_GINT64_FORMAT"ms",
			manager->delay_retry_pending / G_TIME_SPAN_MILLISECOND,
			manager->delay_retry_failed/ G_TIME_SPAN_MILLISECOND);
	GRID_INFO("- fail_pending=%"G_GINT64_FORMAT"ms",
			manager->delay_fail_pending / G_TIME_SPAN_MILLISECOND);
	GRID_INFO("- ping_pending=%"G_GINT64_FORMAT"ms, ping_failed=%"G_GINT64_FORMAT"ms, ping_final=%"G_GINT64_FORMAT"ms",
			manager->delay_ping_pending / G_TIME_SPAN_MILLISECOND,
			manager->delay_ping_failed / G_TIME_SPAN_MILLISECOND,
			manager->delay_ping_final / G_TIME_SPAN_MILLISECOND);
}

GError *
election_manager_create(struct replication_config_s *config,
		struct election_manager_s **result)
{
	EXTRA_ASSERT(result != NULL);
	EXTRA_ASSERT(config != NULL);

	*result = NULL;
	if (NULL == config->get_local_url || NULL == config->get_peers
		|| NULL == config->get_version || ELECTION_MODE_GROUP < config->mode)
		return NEWERROR(ERRCODE_PARAM, "Invalid configuration");

	struct election_manager_s *manager = g_malloc0(sizeof(*manager));
	manager->vtable = &VTABLE;
	manager->delay_wait = SQLX_DELAY_MAXWAIT;
	manager->delay_expire_none = SQLX_DELAY_EXPIRE_NONE;
	manager->delay_expire_final = SQLX_DELAY_EXPIRE_FINAL;
	/* The leader must expire after the slaves or its leaving
	 * will trigger an event on the slaves that will bring it back. */
	if (SQLX_DELAY_EXPIRE_FINAL > 0)
		manager->delay_expire_final_leader =
			(SQLX_DELAY_EXPIRE_FINAL + SQLX_DELAY_PING_FINAL) / 2;
	manager->delay_expire_failed = SQLX_DELAY_EXPIRE_FAILED;
	manager->delay_fail_pending = SQLX_DELAY_MAXIDLE;
	manager->delay_retry_failed = SQLX_DELAY_RESTART_FAILED;
	manager->delay_retry_pending = SQLX_DELAY_ELECTION_REPLAY;
	manager->delay_ping_pending = SQLX_DELAY_PING_PENDING;
	manager->delay_ping_final = SQLX_DELAY_PING_FINAL;
	manager->delay_ping_failed = SQLX_DELAY_PING_FAILED;
	manager->config = config;

	g_mutex_init(&manager->lock);

	manager->members_by_key =
		g_tree_new_full (hashstr_quick_cmpdata, NULL, NULL, NULL);

	manager->conditions =
		g_tree_new_full(hashstr_quick_cmpdata, NULL, g_free, _cond_clean);

	manager->completions =
		g_thread_pool_new(_completion_router, manager, 2, TRUE, NULL);

	manager->synchronous_completions = FALSE;

	*result = manager;
	election_manager_dump_delays(manager);
	return NULL;
}

void
election_manager_set_sync (struct election_manager_s *manager,
		struct sqlx_sync_s *sync)
{
	EXTRA_ASSERT(manager != NULL);
	EXTRA_ASSERT(sync != NULL);
	EXTRA_ASSERT(manager->vtable == &VTABLE);
	manager->sync = sync;
}

void
election_manager_set_peering (struct election_manager_s *manager,
		struct sqlx_peering_s *peering)
{
	EXTRA_ASSERT(manager != NULL);
	EXTRA_ASSERT(peering != NULL);
	EXTRA_ASSERT(manager->vtable == &VTABLE);
	manager->peering = peering;
}

GError *
election_has_peers (struct election_manager_s *m, const struct sqlx_name_s *n,
		gboolean nocache, gboolean *result)
{
	gchar **peers = NULL;
	GError *err = election_get_peers (m, n, nocache, &peers);
	if (err != NULL) {
		*result = FALSE;
		return err;
	}
	*result = peers != NULL && oio_str_is_set(*peers);
	if (peers)
		g_strfreev(peers);
	return NULL;
}

GError *
election_get_peers (struct election_manager_s *m, const struct sqlx_name_s *n,
		gboolean nocache, gchar ***peers)
{
	if (!m) {
		if (peers)
			*peers = g_malloc0(sizeof(void*));
		return NULL;
	}
	return ((struct abstract_election_manager_s*)m)->vtable->election_get_peers
		(m,n,nocache,peers);
}

const char *
election_manager_get_local (const struct election_manager_s *m)
{
	if (!m)
		return NULL;
	return ((struct abstract_election_manager_s*)m)->vtable->get_local(m);
}

enum election_mode_e
election_manager_get_mode (const struct election_manager_s *m)
{
	if (!m)
		return ELECTION_MODE_NONE;
	return ((struct abstract_election_manager_s*)m)->vtable->get_mode(m);
}

static struct election_counts_s
_NOLOCK_count (struct election_manager_s *manager)
{
	struct election_counts_s count = {0};
	count.none = manager->members_by_state[STEP_NONE].count;
	count.pending += manager->members_by_state[STEP_CANDREQ].count;
	count.pending += manager->members_by_state[STEP_CANDOK].count;
	count.pending += manager->members_by_state[STEP_LEAVING].count;
	count.pending += manager->members_by_state[STEP_PRELOST].count;
	count.pending += manager->members_by_state[STEP_PRELEAD].count;
	count.slave = manager->members_by_state[STEP_LOST].count;
	count.master = manager->members_by_state[STEP_LEADER].count;
	count.failed = manager->members_by_state[STEP_FAILED].count;
	count.total = count.none + count.pending + count.master + count.slave + count.failed;
	return count;
}

struct election_counts_s
election_manager_count(struct election_manager_s *manager)
{
	MANAGER_CHECK(manager);
	EXTRA_ASSERT (manager->vtable == &VTABLE);

	_manager_lock(manager);
	struct election_counts_s count = _NOLOCK_count (manager);
	_manager_unlock(manager);
	return count;
}

static GError *
_election_get_peers(struct election_manager_s *manager,
		const struct sqlx_name_s *n, gboolean nocache, gchar ***result)
{
	SQLXNAME_CHECK(n);

	if (!manager || !manager->config || !manager->config->get_peers) {
		if (result)
			*result = NULL;
		return NULL;
	}

	gchar **peers = NULL;
	GError *err = manager->config->get_peers(manager->config->ctx, n, nocache, &peers);
	if (!err) {
		if (result)
			*result = peers;
		else
			g_strfreev (peers);
		return NULL;
	}
	if (peers) {
		g_strfreev (peers);
		peers = NULL;
	}

	g_prefix_error(&err, "get_peers(%s,%s): ", n->base, n->type);
	return err;
}

static void
_manager_clean(struct election_manager_s *manager)
{
	if (!manager)
		return;

	struct election_counts_s count = _NOLOCK_count(manager);
	GRID_DEBUG("%d elections still alive at manager shutdown: %d masters, "
			"%d slaves, %d pending, %d failed, %d exited",
			count.total, count.master, count.slave, count.pending,
			count.failed, count.none);

	if (manager->completions) {
		g_thread_pool_free(manager->completions, FALSE, TRUE);
		manager->completions = NULL;
	}

	if (manager->members_by_key) {
		g_tree_destroy (manager->members_by_key);
		manager->members_by_key = NULL;
	}

	/* Ensure all the items are unlinked */
	for (int i=STEP_NONE; i<STEP_MAX ;++i) {
		struct deque_beacon_s *beacon = manager->members_by_state + i;
		while (beacon->front != NULL) {
			struct election_member_s *m = beacon->front;
			_DEQUE_remove(m);
			m->refcount = 0; /* ugly quirk that cope with an assert on refcount */
			member_destroy (m);
		}
		g_assert (beacon->count == 0);
	}

	if (manager->conditions) {
		g_tree_destroy(manager->conditions);
		manager->conditions = NULL;
	}

	g_mutex_clear(&manager->lock);

	g_free(manager);
}

const char *
_manager_get_local (const struct election_manager_s *manager)
{
	MANAGER_CHECK(manager);
	EXTRA_ASSERT (manager->vtable == &VTABLE);
	if (!manager->config || !manager->config->get_local_url)
		return NULL;
	return manager->config->get_local_url (manager->config->ctx);
}

enum election_mode_e
_manager_get_mode (const struct election_manager_s *manager)
{
	MANAGER_CHECK(manager);
	EXTRA_ASSERT (manager->vtable == &VTABLE);
	if (!manager->config || manager->config->mode <= ELECTION_MODE_NONE)
		return ELECTION_MODE_NONE;
	return manager->config->mode;
}

/* --- Member handling ----------------------------------------------------- */

static req_id_t
manager_next_reqid(struct election_manager_s *m)
{
	return ++ m->next_id;
}

static gboolean
member_has_action(struct election_member_s *m)
{
	return m->pending_GETVERS || m->pending_PIPEFROM || m->requested_PIPEFROM;
}

static const char*
member_get_url(struct election_member_s *m)
{
	return election_manager_get_local(MMANAGER(m));
}

static GError *
member_get_peers(struct election_member_s *m, gboolean nocache, gchar ***peers)
{
	return election_get_peers(MMANAGER(m),
			sqlx_name_mutable_to_const(&m->name), nocache, peers);
}

static void
member_decache_peers(struct election_member_s *m)
{
	GError *err = member_get_peers(m, TRUE, NULL);
	if (err) g_clear_error(&err);
}

static void
member_ref(struct election_member_s *m)
{
	++ m->refcount;
}

static void
member_unref(struct election_member_s *m)
{
	EXTRA_ASSERT (m->refcount > 0);
	-- m->refcount;
}

static GCond*
member_get_cond(struct election_member_s *m)
{
	return m->cond;
}

static GMutex*
member_get_lock(struct election_member_s *m)
{
	return &(MMANAGER(m)->lock);
}

static void
member_lock(struct election_member_s *m)
{
	_manager_lock(m->manager);
}

static void
member_unlock(struct election_member_s *m)
{
	_manager_unlock(m->manager);
}

static void
member_signal(struct election_member_s *m)
{
	g_cond_signal(member_get_cond(m));
}

static void
member_set_master_url(struct election_member_s *m, const char *u)
{
	oio_str_replace(&(m->master_url), u);
	if (u)
		member_debug(__FUNCTION__, "MASTER_URL", m);
}

static void
member_reset_master(struct election_member_s *m)
{
	m->master_id = -1;
	member_set_master_url(m, NULL);
}

static void
member_reset_pending(struct election_member_s *m)
{
	m->reqid_GETVERS = 0;
	m->concurrent_GETVERS = 0;
	m->errors_GETVERS = 0;
	m->pending_GETVERS = 0;
	m->reqid_PIPEFROM = 0;
	m->pending_PIPEFROM = 0;
}

static void
member_reset(struct election_member_s *m)
{
	member_reset_master(m);
	member_reset_pending(m);
	m->myid = -1;
}

static void
member_set_id(struct election_member_s *m, gint64 id)
{
	EXTRA_ASSERT(id >= 0);
	EXTRA_ASSERT(m->myid <= 0);
	m->myid = id;
	member_debug(__FUNCTION__, "ID", m);
}

static void
member_set_status2(struct election_member_s *m, enum election_step_e s,
	   gboolean force_last_status)
{
	_DEQUE_remove (m);
	if (s != m->step || force_last_status)
		m->last_status = oio_ext_monotonic_time();
	m->step = s;
	_DEQUE_add (m);
	if (STATUS_FINAL(s)) {
		member_debug(__FUNCTION__, "FINAL", m);
		member_signal(m);
	} else if (s == STEP_NONE) {
		member_signal(m);
	} /* else ... no need to signal waiting threads unless a final status */
}

static void
member_set_status(struct election_member_s *m, enum election_step_e s)
{
	return member_set_status2(m, s, TRUE);
}

static void
member_set_master_id(struct election_member_s *m, gint64 i64)
{
	EXTRA_ASSERT(i64 >= 0);
	if (i64 != m->master_id)
		member_set_master_url(m, NULL);
	m->master_id = i64;
	member_debug(__FUNCTION__, "MASTER_ID", m);
}

static void
member_log_event(struct election_member_s *member, enum election_step_e pre,
		enum event_type_e evt)
{
	struct logged_event_s *plog;
	plog = member->log + ((member->log_index++) % EVENTLOG_SIZE);
	plog->event = evt;
	plog->pre = pre;
	plog->post = member->step;
}

static void
member_descr(const struct election_member_s *m, gchar *d, gsize ds)
{
	g_snprintf(d, ds,
			"%s %"G_GINT64_FORMAT"/%"G_GINT64_FORMAT"/%s %u %u/%u/%u/%u [%.*s] [%s.%s]",
			_step2str(m->step), m->myid, m->master_id,
			(m->master_url ? m->master_url : ""),
			m->refcount, m->pending_PIPEFROM,
			m->pending_GETVERS, m->errors_GETVERS, m->concurrent_GETVERS,
			(guint) hashstr_len(m->key), hashstr_str(m->key),
			m->name.base, m->name.type);
}

static void
member_trace(const char *func, const char *tag,
		const struct election_member_s *m)
{
	(void) func, (void) tag, (void) m;
#ifdef HAVE_EXTRA_DEBUG
	gchar d[512];
	if (!GRID_TRACE2_ENABLED())
		return;
	member_descr(m, d, sizeof(d));
	GRID_TRACE2("%s %s %s", tag ? tag : "", d, func ? func : "");
#endif
}

static void
member_debug(const char *func, const char *tag,
		const struct election_member_s *m)
{
	gchar d[512];
	member_descr(m, d, sizeof(d));
	GRID_DEBUG("%s %s %s", tag ? tag : "", d, func ? func : "");
}

static void
member_warn(const char *tag, const struct election_member_s *m)
{
	gchar d[512];
	member_descr(m, d, sizeof(d));
	GRID_WARN("%s %s", tag ? tag : "", d);
}

static gchar *
member_fullpath(struct election_member_s *member)
{
	return (member->myid >= 0)
		? g_strdup_printf("%s-%010"G_GINT64_FORMAT,
				hashstr_str(member->key), member->myid)
		: g_strdup_printf("%s-",
				hashstr_str(member->key));
}

static gchar *
member_masterpath(struct election_member_s *member)
{
	return (member->master_id < 0) ? NULL
		: g_strdup_printf("%s-%010"G_GINT64_FORMAT,
				hashstr_str(member->key), member->master_id);
}

static void
member_destroy(struct election_member_s *member)
{
	if (!member)
		return;

	EXTRA_ASSERT (member->refcount == 0);

	member->cond = NULL;
	oio_str_clean (&member->master_url);
	g_free0 (member->key);
	sqlx_name_clean (&member->name);

	memset(member, 0, sizeof(*member));
	member->myid = member->master_id = -1;
	g_free(member);
}

static gboolean
member_group_master(struct election_member_s *member, GArray *i64v)
{
	EXTRA_ASSERT(i64v != NULL);
	EXTRA_ASSERT(i64v->len > 0);
	if (member->myid < 0)
		return FALSE;
	return g_array_index(i64v, gint64, 0) == member->myid;
}

static gboolean
member_in_group(struct election_member_s *member, GArray *i64v)
{
	EXTRA_ASSERT(i64v != NULL);
	if (member->myid < 0)
		return FALSE;
	for (guint i=0; i<i64v->len ;i++) {
		gint64 i64 = g_array_index(i64v, gint64, i);
		if (i64 == member->myid)
			return TRUE;
	}
	return FALSE;
}

static struct election_member_s *
_LOCKED_get_member (struct election_manager_s *ma, const hashstr_t *k)
{
	struct election_member_s *m = g_tree_lookup (ma->members_by_key, k);
	if (m) member_ref (m);
	return m;
}

static GCond *
_manager_get_condition (struct election_manager_s *m, const hashstr_t *k)
{
	GCond *cond = g_tree_lookup (m->conditions, k);
	if (!cond) {
		cond = g_malloc0 (sizeof(GCond));
		g_cond_init (cond);
		g_tree_replace (m->conditions, hashstr_dup(k), cond);
	}
	return cond;
}

static struct election_member_s *
_LOCKED_init_member(struct election_manager_s *manager,
		const struct sqlx_name_s *n, gboolean autocreate)
{
	MANAGER_CHECK(manager);
	NAME_CHECK(n);

	struct hashstr_s *key = sqliterepo_hash_name(n);
	struct election_member_s *member = _LOCKED_get_member (manager, key);
	if (!member && autocreate) {
		member = g_malloc0 (sizeof(*member));
		member->manager = manager;
		member->last_status = oio_ext_monotonic_time ();
		member->key = hashstr_dup(key);
		member->name.base = g_strdup(n->base);
		member->name.type = g_strdup(n->type);
		member->name.ns = g_strdup(n->ns);
		member->myid = member->master_id = -1;
		member->refcount = 2;
		member->cond = _manager_get_condition(manager, key);

		_DEQUE_add (member);
		g_tree_replace(manager->members_by_key, member->key, member);
	}
	g_free(key);
	return member;
}

static struct election_member_s *
manager_get_member (struct election_manager_s *m, const hashstr_t *k)
{
	_manager_lock(m);
	struct election_member_s *member = _LOCKED_get_member (m, k);
	_manager_unlock(m);
	return member;
}

static guint
manager_count_active(struct election_manager_s *manager)
{
	struct election_counts_s count = election_manager_count (manager);
	return count.pending + count.master + count.slave;
}

static gboolean
_run_exit (gpointer k, gpointer v, gpointer i)
{
	(void) k, (void) i;
	struct election_member_s *m = v;
	if (m->step != STEP_NONE && m->step != STEP_LEAVING)
		transition(m, EVT_EXITING, NULL);
	return FALSE;
}

gboolean
election_manager_is_operational(struct election_manager_s *manager)
{
	return (manager != NULL &&
			manager->vtable != NULL &&
			manager->peering != NULL &&
			manager->config != NULL &&
			manager->members_by_key != NULL);
}

void
election_manager_exit_all(struct election_manager_s *manager, gint64 duration,
		gboolean persist)
{
	GRID_INFO("Voluntarily exiting all the elections...");
	MANAGER_CHECK(manager);
	EXTRA_ASSERT (manager->vtable == &VTABLE);
	gint64 pivot = oio_ext_monotonic_time () + duration;

	/* Order the nodes to exit */
	_manager_lock(manager);
	manager->exiting = TRUE;
	g_tree_foreach (manager->members_by_key, _run_exit, NULL);
	_manager_unlock(manager);

	guint count = manager_count_active(manager);
	if (duration <= 0) {
		GRID_INFO("%u elections still active", count);
	} else {
		do {
			GRID_INFO("Waiting for %u active elections", count);
			if (oio_ext_monotonic_time() > pivot) {
				GRID_WARN("TIMEOUT while waiting for active elections");
				break;
			}
			g_usleep(500 * G_TIME_SPAN_MILLISECOND);
		} while ((count = manager_count_active(manager)) > 0);
		if (count == 0)
			GRID_INFO("No more active elections");
	}

	if (!persist)
		manager->exiting = FALSE;
}

#define GS_APPEND_LEN(gs,str) g_string_append_len(gs, str, sizeof(str)-1);

static void
member_json (struct election_member_s *m, GString *gs)
{
	g_string_append (gs, "{");

	/* description */
	g_string_append (gs, "\"local\":{");
	oio_str_gstring_append_json_pair_int (gs, "id", m->myid);
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair (gs, "url", member_get_url(m));
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair (gs, "state", _step2str(m->step));
	g_string_append (gs, "},\"master\":{");
	oio_str_gstring_append_json_pair_int (gs, "id", m->master_id);
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair (gs, "url", m->master_url);
	g_string_append (gs, "},\"base\":{");
	oio_str_gstring_append_json_pair (gs, "name", m->name.base);
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair (gs, "type", m->name.type);
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair (gs, "zk", hashstr_str(m->key));
	g_string_append (gs, "},\"#\":{");
	oio_str_gstring_append_json_pair_int (gs, "R", m->refcount);
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair_int (gs, "P", m->pending_PIPEFROM);
	g_string_append_c (gs, ',');
	oio_str_gstring_append_json_pair_int (gs, "V", m->pending_GETVERS);
	g_string_append_c (gs, '}');

	/* the peers */
	GS_APPEND_LEN(gs, ",\"peers\":");
	gchar **peers = NULL;
	GError *err = member_get_peers(m, FALSE, &peers);
	if (err != NULL) {
		g_string_append_printf(gs, "{\"status\":%d", err->code);
		oio_str_gstring_append_json_pair (gs, "message", err->message);
		g_string_append_c (gs, '}');
	} else if (peers) {
		g_string_append_c (gs, '[');
		for (gchar **p = peers; *p ;p++) {
			if (p!=peers) g_string_append_c(gs, ',');
			oio_str_gstring_append_json_quote(gs, *p);
		}
		g_strfreev(peers);
		g_string_append_c (gs, ']');
	} else {
		GS_APPEND_LEN(gs, "null");
	}

	/* then the livelog */
	GS_APPEND_LEN(gs, ",\"log\":[");
	guint idx = m->log_index - 1;
	for (guint i=0; i<EVENTLOG_SIZE ;i++,idx--) {
		struct logged_event_s *plog = m->log + (idx % EVENTLOG_SIZE);
		if (!plog->pre && !plog->post)
			break;
		if (i!=0)
			g_string_append_c(gs, ',');
		g_string_append_printf(gs, "\"%s:%s:%s\"", _step2str(plog->pre),
				_evt2str(plog->event), _step2str(plog->post));
	}
	g_string_append_c (gs, ']');

	g_string_append_c (gs, '}');
}

void
election_manager_whatabout (struct election_manager_s *m,
		const struct sqlx_name_s *n, gchar *d, gsize ds)
{
	NAME_CHECK(n);
	MANAGER_CHECK(m);
	EXTRA_ASSERT (m->vtable == &VTABLE);
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(ds > 0);

	GString *gs = g_string_sized_new(64);
	hashstr_t *key = sqliterepo_hash_name(n);
	_manager_lock(m);
	struct election_member_s *member = _LOCKED_get_member(m, key);
	if (member) {
		member_json (member, gs);
		member_unref (member);
	} else {
		if (election_manager_get_mode(m) == ELECTION_MODE_NONE)
			g_string_append (gs, "{}");
		else
			g_string_append (gs, "null");
	}
	_manager_unlock (m);

	g_strlcpy (d, gs->str, ds);
	g_string_free (gs, TRUE);
	g_free(key);
}

/* --- Zookeeper callbacks ----------------------------------------------------
 * All of them are called from the zookeeper's thread.
 * We chose to set the election manager in a thread-local slot because ZK
 * contexts for callbackks currently (3.4.6) require that no memory is
 * allocated, especially because of a memory leak on discarded clone watchers.
 * We are forced to pass an integer cast into pointer so that watchers can use
 * them to recover the right election.
 * -------------------------------------------------------------------------- */

static void
step_DeleteRogueNode_completion(int zrc, const void *d)
{
	if (!d)
		return;

	gchar *path = (gchar*) d;

	if (zrc == ZNONODE) {
		GRID_DEBUG("Rogue ZK node at %s disappeared", path);
	} else if (zrc == ZOK) {
		GRID_TRACE("Sucessfully deleted rogue ZK node at %s", path);
	} else if (zrc == ZSESSIONEXPIRED) {
		/* the node will expire, don't flood with logs in this case */
		GRID_DEBUG("Failed to delete rogue ZK node at %s: %s",
				path, zerror(zrc));
	} else {
		GRID_WARN("Failed to delete rogue ZK node at %s: %s",
				path, zerror(zrc));
	}

	g_free(path);
}

/* @private */
enum deferred_action_type_e
{
	DAT_ASKING,
	DAT_LISTING,
	DAT_LEAVING,
	DAT_WATCHING,
	DAT_CREATING,
	DAT_LEFT,
};

/* @private */
struct exec_later_ASKING_context_s
{
	enum deferred_action_type_e magic;
	int zrc;
	struct election_member_s *member;
	gchar master[];
};

static void
exec_later_ASKING_hook(struct exec_later_ASKING_context_s *d)
{
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(DAT_ASKING == d->magic);

	MEMBER_CHECK(d->member);
	member_trace(__FUNCTION__, "DONE", d->member);

	member_lock(d->member);

	if (d->zrc != ZOK)
		transition_error(d->member, EVT_MASTER_KO, d->zrc);
	else {
		if (!d->master[0] || !metautils_url_valid_for_connect(d->master)) {
			transition(d->member, EVT_MASTER_EMPTY, &d->zrc);
		} else {
			const char *myurl = member_get_url(d->member);
			if (strcmp(d->master, myurl) == 0) {
				/* JFS: the supposed master carries our ID (i.e. our URL),
				 * if we accept it as-is, we will create a loop on ourselves.
				 * We delete it and pretend there is no master. */
				gchar *path = member_masterpath(d->member);
				GRID_WARN("Rogue ZK node at %s, triggering deletion", path);
				int zrc2 = sqlx_sync_adelete(d->member->manager->sync, path, -1,
						step_DeleteRogueNode_completion, path);
				if (zrc2 != ZOK) {
					GRID_WARN("Failed! %s", zerror(zrc2));
					g_free(path);
				} // else `path` is freed by the callback

				transition(d->member, EVT_MASTER_EMPTY, &d->zrc);
			} else {
				transition(d->member, EVT_MASTER_OK, d->master);
			}
		}
	}
	member_unref(d->member);
	member_unlock(d->member);
	g_free0 (d);
}

static void
step_AskMaster_completion(int zrc, const char *v, int vlen,
		const struct Stat *s UNUSED, const void *d)
{
	if (vlen > 256)
		vlen = 0;
	struct exec_later_ASKING_context_s *ctx = g_malloc0(sizeof(*ctx) + vlen + 1);
	ctx->magic = DAT_ASKING;
	ctx->zrc = zrc;
	ctx->member = (struct election_member_s*) d;
	if (vlen)
		memcpy(ctx->master, v, vlen);

	struct election_manager_s *M = ctx->member->manager;
	if (M->synchronous_completions) {
		return exec_later_ASKING_hook(ctx);
	} else {
		gboolean rc = g_thread_pool_push(ctx->member->manager->completions, ctx, NULL);
		g_assert_true(rc);
	}
}

/* @private */
struct exec_later_LISTING_context_s
{
	enum deferred_action_type_e magic;
	int zrc;
	struct election_member_s *member;
	GArray *i64v;
};

static void
exec_later_LISTING_hook (struct exec_later_LISTING_context_s *d)
{
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(DAT_LISTING == d->magic);
	MEMBER_CHECK(d->member);
	member_trace(__FUNCTION__, "DONE", d->member);

	member_lock(d->member);
	if (d->zrc != ZOK)
		transition_error(d->member, EVT_LIST_KO, d->zrc);
	else
		transition(d->member, EVT_LIST_OK, d->i64v);
	member_unref(d->member);
	member_unlock(d->member);

	g_array_free(d->i64v, TRUE);
	g_free(d);
}

static void
step_ListGroup_completion(int zrc, const struct String_vector *sv,
		const void *data)
{
	struct exec_later_LISTING_context_s *ctx = g_malloc0(sizeof(*ctx));
	ctx->magic = DAT_LISTING;
	ctx->zrc = zrc;
	ctx->member = (struct election_member_s*) data;
	ctx->i64v = nodev_to_int64v(sv, hashstr_str(ctx->member->key));

	struct election_manager_s *M = ctx->member->manager;
	if (M->synchronous_completions) {
		return exec_later_LISTING_hook(ctx);
	} else {
		gboolean rc = g_thread_pool_push(ctx->member->manager->completions, ctx, NULL);
		g_assert_true(rc);
	}
}

/* @private */
struct exec_later_LEAVING_context_s
{
	enum deferred_action_type_e magic;
	int zrc;
	struct election_member_s *member;
};

static void
exec_later_LEAVING_hook(struct exec_later_LEAVING_context_s *d)
{
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(DAT_LEAVING == d->magic);
	MEMBER_CHECK(d->member);
	member_trace(__FUNCTION__, "DONE", d->member);

	member_lock(d->member);
	if (d->zrc == ZNONODE)
		transition(d->member, EVT_LEAVE_OK, NULL);
	else if (d->zrc != ZOK)
		transition_error(d->member, EVT_LEAVE_KO, d->zrc);
	else
		transition(d->member, EVT_LEAVE_OK, NULL);
	member_unref(d->member);
	member_unlock(d->member);

	g_free(d);
}

static void
step_LeaveElection_completion(int zrc, const void *d)
{
	struct exec_later_LEAVING_context_s *ctx = g_malloc0(sizeof(*ctx));
	ctx->magic = DAT_LEAVING;
	ctx->zrc = zrc;
	ctx->member = (struct election_member_s*) d;

	struct election_manager_s *M = ctx->member->manager;
	if (M->synchronous_completions) {
		return exec_later_LEAVING_hook(ctx);
	} else {
		gboolean rc = g_thread_pool_push(ctx->member->manager->completions, ctx, NULL);
		g_assert_true(rc);
	}
}

/* @private */
struct exec_later_WATCHING_context_s
{
	enum deferred_action_type_e magic;
	int zrc;
	struct election_member_s *member;
};

static void
exec_later_WATCHING_hook(struct exec_later_WATCHING_context_s *d)
{
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(DAT_WATCHING == d->magic);
	MEMBER_CHECK(d->member);

	member_lock(d->member);
	member_trace(__FUNCTION__, "DONE", d->member);
	if (d->zrc == ZNONODE)
		transition(d->member, EVT_NODE_LEFT, &d->zrc);
	else {
		if (d->zrc != ZOK)
			transition_error(d->member, EVT_EXITING, d->zrc);
	}
	member_unref(d->member);
	member_unlock(d->member);
	g_free(d);
}

static void
step_WatchNode_completion(int zrc, const struct Stat *s UNUSED, const void *d)
{
	struct exec_later_WATCHING_context_s *ctx = g_malloc0(sizeof(*ctx));
	ctx->magic = DAT_WATCHING;
	ctx->zrc = zrc;
	ctx->member = (struct election_member_s*) d;

	struct election_manager_s *M = ctx->member->manager;
	if (M->synchronous_completions) {
		return exec_later_WATCHING_hook(ctx);
	} else {
		gboolean rc = g_thread_pool_push(ctx->member->manager->completions, ctx, NULL);
		g_assert_true(rc);
	}
}

/* @private */
struct exec_later_CREATING_context_s
{
	enum deferred_action_type_e magic;
	int zrc;
	struct election_member_s *member;
	gint64 local_id;
};

static void
member_warn_failed_creation(struct election_member_s *member, int zrc)
{
	gchar *p = member_fullpath(member);
	GRID_WARN("CREATE failed [%s.%s] [%s] : (%d) %s",
			member->name.base, member->name.type, p, zrc, zerror(zrc));
	g_free(p);
}

static void
exec_later_CREATING_hook(struct exec_later_CREATING_context_s *d)
{
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(DAT_CREATING == d->magic);
	MEMBER_CHECK(d->member);

	member_lock(d->member);
	member_trace(__FUNCTION__, "DONE", d->member);
	_thlocal_set_manager (d->member->manager);

	if (d->zrc != ZOK) {
		member_warn_failed_creation(d->member, d->zrc);
		transition_error(d->member, EVT_CREATE_KO, d->zrc);
	} else {
		if (d->local_id < 0) {
			member_warn_failed_creation(d->member, d->zrc);
			transition(d->member, EVT_CREATE_KO, NULL);
		} else {
			transition(d->member, EVT_CREATE_OK, &d->local_id);
		}
	}
	member_unref(d->member);
	member_unlock(d->member);

	g_free(d);
}

static void
step_StartElection_completion(int zrc, const char *path, const void *d)
{
	struct exec_later_CREATING_context_s *ctx = g_malloc0(sizeof(*ctx));
	ctx->magic = DAT_CREATING;
	ctx->member = (struct election_member_s *) d;
	if (path) {
		ctx->local_id = g_ascii_strtoll(strrchr(path, '-')+1, NULL, 10);
		ctx->zrc = zrc;
	} else {
		ctx->local_id = -1;
		ctx->zrc = ZNONODE;
	}

	struct election_manager_s *M = ctx->member->manager;
	_thlocal_set_manager(M);
	if (M->synchronous_completions) {
		return exec_later_CREATING_hook(ctx);
	} else {
		gboolean rc = g_thread_pool_push(ctx->member->manager->completions, ctx, NULL);
		g_assert_true(rc);
	}
}

/* @private */
struct deferred_watcher_context_s
{
	enum deferred_action_type_e magic;
	int type;
	int state;
	guint gen;
	enum event_type_e evt;
	char path[];
};

static struct election_member_s *
_find_member (struct election_manager_s *M, const char *path)
{
	if (!M) return NULL;

	const char *slash = strrchr(path, '/');
	if (!slash) return NULL;
	slash ++;

	const char *stripe = strchr(slash, '-');
	if (!stripe) return NULL;

	const size_t len = stripe - slash;
	hashstr_t *hkey = NULL;
	HASHSTR_ALLOCA_LEN(hkey, slash, len);

	struct election_member_s *member = NULL;
	_manager_lock(M);
	member = _LOCKED_get_member(M, hkey);
	if (member) {
		return member;
	} else {
		GRID_WARN("watcher: [%s] no election found", hashstr_str(hkey));
	}
	_manager_unlock(M);
	return NULL;
}

static void
_deferred_watcher_hook(struct deferred_watcher_context_s *d,
		struct election_manager_s *M)
{
	EXTRA_ASSERT(d != NULL);
	EXTRA_ASSERT(M != NULL);
	EXTRA_ASSERT(DAT_LEFT == d->magic);

	if (d->type == ZOO_SESSION_EVENT) {
		/* Big disconnection ... let's expire everything ! */
		GRID_DEBUG("ZK DISCONNECTED, expiring");
		guint count = 0;
		_manager_lock(M);
		for (int i=STEP_CANDREQ; i<STEP_MAX ;++i) {
			struct deque_beacon_s *b = M->members_by_state + i;
			while (b->front != NULL) {
				struct election_member_s *m = b->front;
				member_reset(m);
				member_set_status(m, STEP_NONE);
				count ++;
			}
		}
		_manager_unlock(M);
		/* All the items are moved in the STEP_NONE list, so subsequent
		 * calls shouldn't iterate on anything */
		if (count)
			GRID_WARN("ZK DISCONNECTED, expired %u", count);
	} else if (d->type == ZOO_DELETED_EVENT) {
		struct election_member_s *member = _find_member(M, d->path);
		if (NULL != member) {
			transition(member, d->evt, NULL);
			member_unref(member);
			member_unlock(member);
		}
	}

	g_free(d);
}

static void
_watcher_change(const int type, const int state,
		const char *path, void *d, const int evt)
{
	if (type != ZOO_SESSION_EVENT && type != ZOO_DELETED_EVENT)
		return;
	if (type == ZOO_SESSION_EVENT &&
			state != ZOO_EXPIRED_SESSION_STATE &&
			state != ZOO_AUTH_FAILED_STATE)
		return;

	const char *slash = path ? strrchr(path, '/') : NULL;
	const size_t len = slash ? strlen(slash) : 0;

	struct deferred_watcher_context_s *ctx = g_malloc0(sizeof(*ctx) + len + 1);
	ctx->magic = DAT_LEFT;
	ctx->type = type;
	ctx->state = state;
	ctx->gen = GPOINTER_TO_UINT(d);
	ctx->evt = evt;
	if (slash && len)
		memcpy(ctx->path, slash, len);

	struct election_manager_s *M = _thlocal_get_manager();
	if (M->synchronous_completions) {
		return _deferred_watcher_hook(ctx, M);
	} else {
		gboolean rc = g_thread_pool_push(M->completions, ctx, NULL);
		g_assert_true(rc);
	}
}

static void
step_WatchMaster_change(zhandle_t *handle UNUSED, int type, int state,
		const char *path, void *d)
{
	return _watcher_change(type, state, path, d, EVT_MASTER_CHANGE);
}

static void
step_WatchNode_change(zhandle_t *handle UNUSED, int type, int state,
		const char *path, void *d)
{
	return _watcher_change(type, state, path, d, EVT_NODE_LEFT);
}

static void
_completion_router(gpointer p, gpointer u)
{
	switch (*((enum deferred_action_type_e*)p)) {
		case DAT_CREATING:
			return exec_later_CREATING_hook(p);
		case DAT_ASKING:
			return exec_later_ASKING_hook(p);
		case DAT_LISTING:
			return exec_later_LISTING_hook(p);
		case DAT_LEAVING:
			return exec_later_LEAVING_hook(p);
		case DAT_WATCHING:
			return exec_later_WATCHING_hook(p);
		case DAT_LEFT:
			return _deferred_watcher_hook(p, u);
	}
	g_assert_not_reached();
}

/* ------------------------------------------------------------------------- */

static int
step_StartElection_start(struct election_member_s *member)
{
	member_trace(__FUNCTION__, "ACTION", member);

	const char *myurl = member_get_url(member);
	gchar *path = member_fullpath(member);
	int zrc = sqlx_sync_acreate(member->manager->sync,
			path, myurl, strlen(myurl), ZOO_EPHEMERAL|ZOO_SEQUENCE,
			step_StartElection_completion, member);
	g_free(path);

	if (zrc == ZOK)
		member_ref(member);
	return zrc;
}

static int
step_AskMaster_start(struct election_member_s *member)
{
	int zrc = ZNONODE;
	gchar *path = NULL;

	member_trace(__FUNCTION__, "ACTION", member);

	if (NULL != (path = member_masterpath(member))) {
		zrc = sqlx_sync_awget(member->manager->sync, path,
				step_WatchMaster_change, GUINT_TO_POINTER(0),
				step_AskMaster_completion, member);
		g_free(path);
		if (zrc == ZOK)
			member_ref(member);
	}

	return zrc;
}

static int
step_WatchNode_start(struct election_member_s *member)
{
	member_trace(__FUNCTION__, "ACTION", member);

	gchar *path = member_fullpath(member);
	int zrc = sqlx_sync_awexists(member->manager->sync, path,
			step_WatchNode_change, GUINT_TO_POINTER(0),
			step_WatchNode_completion, member);
	g_free(path);
	if (zrc == ZOK)
		member_ref(member);
	return zrc;
}

static int
step_ListGroup_start(struct election_member_s *member)
{
	member_trace(__FUNCTION__, "ACTION", member);

	gchar *path = member_fullpath(member);
	int zrc = sqlx_sync_awget_siblings(member->manager->sync, path,
			NULL, NULL,
			step_ListGroup_completion, member);
	g_free(path);
	if (zrc == ZOK)
		member_ref(member);
	return zrc;
}

static int
step_LeaveElection_start(struct election_member_s *member)
{
	member_trace(__FUNCTION__, "ACTION", member);

	gchar *path  = member_fullpath(member);
	int zrc = sqlx_sync_adelete(member->manager->sync, path, -1,
			step_LeaveElection_completion, member);
	g_free(path);
	if (zrc == ZOK)
		member_ref(member);
	return zrc;
}

/* ------------------------------------------------------------------------- */

enum election_op_e {
	ELOP_NONE, ELOP_START, ELOP_RESYNC, ELOP_EXIT
};

static GError *
_election_make(struct election_manager_s *m, const struct sqlx_name_s *n,
		enum election_op_e op)
{
	MANAGER_CHECK(m);
	SQLXNAME_CHECK(n);

	if (op != ELOP_EXIT) {
		gboolean peers_present = FALSE;
		GError *err = election_has_peers(m, n, FALSE, &peers_present);
		if (err != NULL) {
			g_prefix_error(&err, "Election error: ");
			return err;
		}
		if (!peers_present) {
			GRID_DEBUG("No peer for [%s][%s]", n->base, n->type);
			return NULL;
		}
	}

	_manager_lock(m);
	struct election_member_s *member = _LOCKED_init_member(m, n, op != ELOP_EXIT);
	switch (op) {
		case ELOP_NONE:
			member->last_atime = oio_ext_monotonic_time ();
			break;
		case ELOP_START:
			member->last_atime = oio_ext_monotonic_time ();
			transition(member, EVT_NONE, NULL);
			break;
		case ELOP_RESYNC:
			member->last_atime = oio_ext_monotonic_time ();
			transition(member, EVT_RESYNC_REQ, NULL);
			break;
		case ELOP_EXIT:
			if (member)
				transition(member, EVT_EXITING, NULL);
			break;
	}
	if (member)
		member_unref(member);
	_manager_unlock(m);

	return NULL;
}

static GError *
_election_trigger_RESYNC(struct election_manager_s *manager,
		const struct sqlx_name_s *n)
{
	return _election_make(manager, n, ELOP_RESYNC);
}

static GError *
_election_init(struct election_manager_s *manager, const struct sqlx_name_s *n)
{
	return _election_make(manager, n, ELOP_NONE);
}

static GError *
_election_start(struct election_manager_s *manager, const struct sqlx_name_s *n)
{
	return _election_make(manager, n, ELOP_START);
}

static GError *
_election_exit(struct election_manager_s *manager, const struct sqlx_name_s *n)
{
	return _election_make(manager, n, ELOP_EXIT);
}

static gboolean
wait_for_final_status(struct election_member_s *m, gint64 deadline)
{
	while (!STATUS_FINAL(m->step)) {

		const gint64 now = oio_ext_monotonic_time();
		m->last_atime = now;
		transition(m, EVT_NONE, NULL);

		/* compare internal timers to our fake'able clock */
		if (now > deadline) {
			GRID_WARN("TIMEOUT! (waiting for election status) [%s.%s]",
					m->name.base, m->name.type);
			return FALSE;
		}
		if (ACTION_FAIL == _member_get_next_action(m)) {
			GRID_WARN("TIMEOUT! (election pending for too long) [%s.%s]",
					m->name.base, m->name.type);
			return FALSE;
		}

		GRID_TRACE("Still waiting for a final status on [%s.%s]",
				m->name.base, m->name.type);

		/* perform the real WAIT on the real clock. */
		m->manager->when_lock = 0;
		g_cond_wait_until(member_get_cond(m), member_get_lock(m),
				g_get_monotonic_time() + oio_election_period_cond_wait);
		m->manager->when_lock = g_get_monotonic_time();
	}

	m->last_atime = oio_ext_monotonic_time ();
	return TRUE;
}

static enum election_status_e
_election_get_status(struct election_manager_s *mgr,
		const struct sqlx_name_s *n, gchar **master_url)
{
	int rc;
	gchar *url = NULL;

	MANAGER_CHECK(mgr);
	EXTRA_ASSERT(n != NULL);

	gint64 deadline = oio_ext_monotonic_time () + mgr->delay_wait;

	_manager_lock(mgr);
	struct election_member_s *m = _LOCKED_init_member(mgr, n, TRUE);

	if (!wait_for_final_status(m, deadline)) // TIMEOUT!
		rc = STEP_FAILED;
	else {
		rc = m->step;
		if (rc == STEP_LOST) {
			if (m->master_url)
				url = g_strdup(m->master_url);
		}
	}

	member_unref(m);
	if (rc == STEP_NONE || STATUS_FINAL(rc))
		member_signal(m);
	_manager_unlock(mgr);

	GRID_TRACE("STEP=%s/%d master=%s", _step2str(rc), rc, url);
	switch (rc) {
		case STEP_LEADER:
			return ELECTION_LEADER;
		case STEP_LOST:
			if (master_url)
				*master_url = url;
			else
				g_free(url);
			url = NULL;
			return ELECTION_LOST;
		default:
			return ELECTION_FAILED;
	}
}

/* ------------------------------------------------------------------------- */

static gboolean
defer_USE(struct election_member_s *member)
{
	const gint64 now = oio_ext_monotonic_time();

	/* Sometimes, defer_USE() is called after a check for a delay (based on
	 * last_USE), sometimes not. When there is already a check, the delay is
	 * ~ always longer than the following G_TIME_SPAN_SECOND, so this check
	 * is harmless.
	 * However, having a hard limit to a minimum of 1s between 2 USE for the
	 * same election is a good thing, IMO (jfs). */
	if ((now - member->last_USE) < G_TIME_SPAN_SECOND) {
		member_trace(__FUNCTION__, "USE avoided (too recent)", member);
		return TRUE;
	}

	gchar **peers = NULL;
	GError *err = member_get_peers(member, FALSE, &peers);
	if (err != NULL) {
		GRID_WARN("[%s] Election initiated (%s) but get_peers error: (%d) %s",
				__FUNCTION__, _step2str(member->step), err->code, err->message);
		g_clear_error(&err);
		return FALSE;
	}

	if (!peers || !*peers) {
		member_trace(__FUNCTION__, "USE avoided (no peers)", member);
	} else {
		member->last_USE = oio_ext_monotonic_time();
		for (gchar **p = peers; peers && *p; p++) {
			sqlx_peering__use (member->manager->peering, *p,
					sqlx_name_mutable_to_const(&member->name));
		}
		member_trace(__FUNCTION__, "USE scheduled", member);
	}

	if (peers) g_strfreev(peers);
	return TRUE;
}

static void
_result_PIPEFROM (GError *e, struct election_manager_s *manager,
		const struct sqlx_name_s *n, guint reqid)
{
	hashstr_t *key = sqliterepo_hash_name (n);

	if (!e || CODE_IS_OK(e->code)) {
		GRID_DEBUG("PIPEFROM ok [%s.%s] [%s]",
				n->base, n->type, hashstr_str(key));
	} else {
		GRID_WARN("PIPEFROM failed [%s.%s] [%s]: (%d) %s",
				n->base, n->type, hashstr_str(key),
				e->code, e->message);
	}

	struct election_member_s *member = manager_get_member (manager, key);
	g_free (key);

	if (member) {
		member_lock(member);
		/* We do the transition even if we undergo an error.
		 * This means we are not consistent but eventually consistent. */
		transition(member, EVT_RESYNC_DONE, &reqid);
		member_unref(member);
		member_unlock(member);
	}
}

static void
defer_PIPEFROM(struct election_member_s *member)
{
	MEMBER_CHECK(member);

	const char *source = member->master_url;
	const char *target = member_get_url(member);

	EXTRA_ASSERT(target != NULL);
	EXTRA_ASSERT(source != NULL);

	if (member->pending_PIPEFROM)
		member_debug(__FUNCTION__, "PIPEFROM avoided", member);
	else {
		member->reqid_PIPEFROM = manager_next_reqid(member->manager);
		member->requested_PIPEFROM = 0;
		member->pending_PIPEFROM = 1;

		sqlx_peering__pipefrom (member->manager->peering, target,
				sqlx_name_mutable_to_const(&member->name), source,
				member->manager, member->reqid_PIPEFROM, _result_PIPEFROM);

		member_debug(__FUNCTION__, "PIPEFROM scheduled", member);
	}
}

static void
_result_GETVERS (GError *enet,
		struct election_manager_s *manager, const struct sqlx_name_s *name,
		guint reqid, GTree *vremote)
{
	GError *err = NULL;
	GTree *vlocal = NULL;

	EXTRA_ASSERT(manager != NULL);
	EXTRA_ASSERT(name != NULL);
	EXTRA_ASSERT((enet != NULL) ^ (vremote != NULL));

	if (enet) {
		err = g_error_copy(enet);
	} else {
		err = manager->config->get_version (manager->config->ctx, name, &vlocal);
		EXTRA_ASSERT ((err != NULL) ^ (vlocal != NULL));
	}

	if (!err) {
		gint64 worst = 0;
		err = version_validate_diff(vlocal, vremote, &worst);
		if (NULL != err) {
			if (err->code == CODE_PIPETO) {
				GRID_DEBUG("Remote outdated : (%d) %s",
						err->code, err->message);
				g_clear_error(&err);
			}
		} else {
			if (worst < 0)
				err = NEWERROR(CODE_PIPEFROM, "One diff missed");
		}
	}

	hashstr_t *key = sqliterepo_hash_name (name);
	struct election_member_s *member = manager_get_member(manager, key);
	if (!member) {
		GRID_WARN("GETVERS Election disappeared [%s]", hashstr_str(key));
	} else {

		MEMBER_CHECK(member);

		member_lock(member);
		if (!err)
			transition(member, EVT_GETVERS_OK, &reqid);
		else if (err->code == CODE_PIPEFROM)
			transition(member, EVT_GETVERS_OUTDATED, &reqid);
		else if (err->code == CODE_CONCURRENT)
			transition(member, EVT_GETVERS_CONCURRENT, &reqid);
		else {
			GRID_DEBUG("GETVERS error : (%d) %s", err->code, err->message);
			if (err->code == CODE_CONTAINER_NOTFOUND) {
				// We may have asked the wrong peer
				member_decache_peers(member);
			}
			transition(member, EVT_GETVERS_ERROR, &reqid);
		}
		member_unlock(member);
	}

	if (err) g_clear_error(&err);
	if (vlocal) g_tree_destroy(vlocal);
	g_free (key);
}

static void
defer_GETVERS(struct election_member_s *member)
{
	MEMBER_CHECK(member);
	GRID_TRACE2("%s(%p)", __FUNCTION__, member);

	gchar **peers = NULL;
	GError *err = member_get_peers(member, FALSE, &peers);
	if (err != NULL) {
		GRID_WARN("[%s] Election initiated (%s) but get_peers error: (%d) %s",
				__FUNCTION__, _step2str(member->step), err->code, err->message);
		g_clear_error(&err);
		return;
	}

	if (member->pending_GETVERS > 0)
		member_debug(__FUNCTION__ , "GETVERS lost", member);

	const guint pending = peers ? g_strv_length(peers) : 0;
	member->concurrent_GETVERS = 0;
	member->errors_GETVERS = 0;
	member->pending_GETVERS = pending;
	member->reqid_GETVERS = manager_next_reqid(member->manager);

	for (gchar **p = peers; peers && *p; p++)
		sqlx_peering__getvers (member->manager->peering, *p,
				sqlx_name_mutable_to_const(&member->name), member->manager,
				member->reqid_GETVERS, _result_GETVERS);

	member_trace(__FUNCTION__ , "GETVERS scheduled", member);
	g_strfreev(peers);
}

/* -------------------------------------------------------------------------- */

static void
member_ask_RESYNC_if_not_pending(struct election_member_s *member)
{
	if (!member->pending_PIPEFROM)
		member->requested_PIPEFROM = 1;
}

static void
become_failed (struct election_member_s *member)
{
	/* setting last_USE to now avoids sending USE as soon as arrived in
	 * the set of FAILED elections. */
	member->last_USE = oio_ext_monotonic_time ();
	member_set_status(member, STEP_FAILED);
}

static void
become_leaver(struct election_member_s *member)
{
	member->delayed_NODE_LEFT = 0;
	member->delayed_MASTER_CHANGE = 0;

	member_reset_master(member);
	member_reset_pending(member);
	/* Update status time only if we actually change status.
	 * See STEP_LEAVING case in _member_get_next_action(). */
	member_set_status2(member, STEP_LEAVING, FALSE);
	int zrc = step_LeaveElection_start(member);
	if (zrc != ZOK)
		become_failed (member);
}

static void
become_candidate(struct election_member_s *member)
{
	member->delayed_NODE_LEFT = 0;
	member->delayed_MASTER_CHANGE = 0;

	member_reset_master(member);
	member_set_status2(member, STEP_CANDOK, FALSE);
	if (ZOK != step_ListGroup_start(member))
		become_failed (member);
}

static void
restart_election(struct election_member_s *member)
{
	if (member->myid > 0)
		return become_leaver(member);

	member_reset(member);
	if (member->manager->exiting)
		return member_set_status(member, STEP_NONE);

	member->requested_USE = 0;

	member_set_status(member, STEP_CANDREQ);
	if (!defer_USE(member))
		return become_failed (member);

	int zrc = step_StartElection_start(member);
	if (ZOK != zrc) {
		member_warn_failed_creation(member, zrc);
		return become_failed (member);
	}
}

static gboolean
member_concerned_by_GETVERS(struct election_member_s *member, guint *reqid)
{
	if (*reqid > member->reqid_GETVERS) {
		GRID_WARN("GETVERS replied from the future! (expected %u, got %u)",
				member->reqid_GETVERS, *reqid);
		return FALSE;
	} else if (*reqid < member->reqid_GETVERS) {
		GRID_WARN("GETVERS replied from the past! (expected %u, got %u)",
				member->reqid_GETVERS, *reqid);
		return FALSE;
	}
	if (member->pending_GETVERS > 0) {
		if (! -- member->pending_GETVERS)
			member->reqid_GETVERS = 0;
	}
	return TRUE;
}

static void
member_finish_PRELOST(struct election_member_s *member)
{
	if (member_has_action(member))
		return;

	const int errors = member->errors_GETVERS;
	const int concurrent = member->concurrent_GETVERS;
	const int node_left = member->delayed_NODE_LEFT;
	const int master_change = member->delayed_MASTER_CHANGE;

	member->errors_GETVERS = 0;
	member->concurrent_GETVERS = 0;
	member->delayed_NODE_LEFT = 0;
	member->delayed_MASTER_CHANGE = 0;

	if (node_left)
		return become_leaver(member);
	if (master_change)
		return become_candidate(member);

	// FIXME compare to the sizeof of the quorum
	if (errors > 1)
		return become_leaver(member);
	if (concurrent > 1)
		return member_ask_RESYNC_if_not_pending(member);

	if (member->master_url)
		member_set_status(member, STEP_LOST);
}

static void
member_finish_PRELEAD(struct election_member_s *member)
{
	if (member_has_action(member))
		return;

	const int errors = member->errors_GETVERS;
	const int concurrent = member->concurrent_GETVERS;
	const int node_left = member->delayed_NODE_LEFT;

	member->errors_GETVERS = 0;
	member->concurrent_GETVERS = 0;
	member->delayed_NODE_LEFT = 0;
	member->delayed_MASTER_CHANGE = 0;

	if (node_left)
		return become_leaver(member);

	// FIXME compare to the sizeof of the quorum
	if (errors > 1)
		return become_leaver(member);
	if (concurrent > 1)
		return become_leaver(member);

	return member_set_status(member, STEP_LEADER);
}

static enum sqlx_action_e
_member_get_next_action (const struct election_member_s *m)
{
	const struct election_manager_s *M = m->manager;

	switch (m->step) {

		case STEP_NONE:
			if (_is_over(m->last_status, M->delay_expire_none))
				return ACTION_EXPIRE;
			return ACTION_NONE;

		case STEP_LEAVING:
			/* ACTION_LEAVE did not get us out of STEP_LEAVING,
			 * go to STEP_FAILED. */
			if (_is_over (m->last_status, M->delay_fail_pending))
				return ACTION_FAIL;
			/* Retry the deletion of our own node. We must ensure that
			 * ACTION_LEAVE does not update last_status or we may never
			 * reach delay_fail_pending and return ACTION_FAIL. */
			if (_is_over (m->last_status, M->delay_retry_pending))
				return ACTION_LEAVE;
			return ACTION_NONE;

		case STEP_CANDREQ:
			if (_is_over(m->last_status, M->delay_fail_pending))
				return ACTION_FAIL;
			/* There are very few chances that STEP_CANDREQ fails to move
			 * forward. The only way to retry is to restart election. */
			if (_is_over(m->last_status, M->delay_retry_pending))
				return ACTION_RESTART;
			return ACTION_NONE;

		case STEP_CANDOK:
			if (_is_over(m->last_status, M->delay_fail_pending))
				return ACTION_FAIL;
			if (_is_over(m->last_status, M->delay_retry_pending))
				return ACTION_RETRY;
			return ACTION_NONE;

		case STEP_PRELOST:
		case STEP_PRELEAD:
			if (_is_over (m->last_status, M->delay_fail_pending))
				return ACTION_FAIL;
			/* ACTION_RETRY (calling defer_GETVERS()) is useful for both
			 * STEP_PRELEAD and STEP_PRELOST, but for the latter we may
			 * need to call step_AskMaster_start() again.
			 * TODO: split STEP_PRELOST and add STEP_MASTER_KNOWN */
			if (_is_over (m->last_status, M->delay_retry_pending))
				return ACTION_RETRY;
			if (_is_over (m->last_USE, M->delay_ping_pending))
				return ACTION_PING;
			return ACTION_NONE;

		case STEP_LEADER:
			if (_is_over (m->last_atime, M->delay_expire_final_leader))
				return ACTION_LEAVE;
			if (_is_over (m->last_USE, M->delay_ping_final))
				return ACTION_PING;
			return ACTION_NONE;

		case STEP_LOST:
			if (_is_over (m->last_atime, M->delay_expire_final))
				return ACTION_LEAVE;
			if (_is_over (m->last_USE, M->delay_ping_final))
				return ACTION_PING;
			return ACTION_NONE;

		case STEP_FAILED:
			if (_is_over (m->last_atime, M->delay_expire_failed))
				return ACTION_LEAVE;
			if (_is_over (m->last_status, M->delay_retry_failed))
				return ACTION_RESTART;
			if (_is_over (m->last_USE, M->delay_ping_failed))
				return ACTION_PING;
			return ACTION_NONE;
	}

	g_assert_not_reached ();
	return ACTION_NONE;
}

static void
_member_react (struct election_member_s *member,
		enum event_type_e evt,
		void *evt_arg)
{
	GArray *i64v = NULL;
	guint reqid;
	int zrc;

	MEMBER_CHECK(member);
	if (GRID_DEBUG_ENABLED()) {
		gchar tag[256] = "";
		g_snprintf(tag, sizeof(tag), "EVENT:%s", _evt2str(evt));
		member_debug(__FUNCTION__, tag, member);
	}

	switch (evt) {

		case EVT_DISCONNECTED:
			member_reset(member);
			return become_failed (member);

		case EVT_RESYNC_DONE:
			EXTRA_ASSERT(evt_arg != NULL);
			reqid = *((guint*)evt_arg);
			if (reqid > member->reqid_PIPEFROM) {
				GRID_WARN("PIPEFROM replied from the future! "
						"(expected reqid %u, got %u)",
						member->reqid_PIPEFROM, reqid);
				member->requested_PIPEFROM = 1;
			} else if (reqid < member->reqid_PIPEFROM) {
				GRID_WARN("PIPEFROM replied from the past! "
						"(expected reqid %u, got %u)",
						member->reqid_PIPEFROM, reqid);
				member->requested_PIPEFROM = 1;
			}
			member->pending_PIPEFROM = 0;
			member->reqid_PIPEFROM = 0;
			break;

		default:
			break;
	}

	switch (member->step) {

		case STEP_NONE:
			EXTRA_ASSERT(member->myid == -1);
			EXTRA_ASSERT(member->master_id == -1);
			EXTRA_ASSERT(member->master_url == NULL);
			EXTRA_ASSERT(member->pending_watch == 0);
			EXTRA_ASSERT(member->pending_PIPEFROM == 0);
			switch (evt) {
				case EVT_NONE:
					member->requested_USE = 0;
					if (member->manager->exiting) {
						member_reset_pending (member);
						return;
					}
					if (!defer_USE(member))
						return become_failed (member);
					zrc = step_StartElection_start(member);
					if (ZOK != zrc) {
						member_warn_failed_creation(member, zrc);
						return become_failed (member);
					}
					return member_set_status(member, STEP_CANDREQ);
				case EVT_RESYNC_REQ:
					member->requested_PIPEFROM = 1;
					member->requested_USE = 1;
					return;
				case EVT_EXITING:
					return;
				case EVT_CREATE_OK:
					member_warn("ABNORMAL", member);
					/* FALLTHROUGH */
				default:
					GRID_DEBUG("IGNORED");
					return;
			}

		case STEP_CANDREQ:
			EXTRA_ASSERT(member->myid == -1);
			EXTRA_ASSERT(member->master_id == -1);
			EXTRA_ASSERT(member->master_url == NULL);
			EXTRA_ASSERT(member->pending_watch == 0);
			EXTRA_ASSERT(member->pending_PIPEFROM == 0);
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_RESYNC_REQ:
					member->requested_PIPEFROM = 1;
					return;
				case EVT_EXITING:
					member->requested_EXIT = 1;
					return;
				case EVT_CREATE_OK:
					EXTRA_ASSERT(evt_arg != NULL);
					member_set_id(member, *((gint64*)evt_arg));
					if (ZOK != step_WatchNode_start(member))
						return become_failed (member);
					return become_candidate (member);
				case EVT_CREATE_KO:
					return become_failed(member);
				default:
					GRID_DEBUG("IGNORED");
					return;
			}
			return;

		case STEP_CANDOK:
			//EXTRA_ASSERT(member->myid > 0);
			EXTRA_ASSERT(member->master_id == -1);
			EXTRA_ASSERT(member->master_url == NULL);
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_RESYNC_REQ:
					member_ask_RESYNC_if_not_pending (member);
					return;
				case EVT_EXITING:
					return become_leaver(member);
				case EVT_LIST_OK:
					i64v = evt_arg;
					EXTRA_ASSERT(i64v != NULL);
					if (!member_in_group(member, i64v)) {
						member_trace(__FUNCTION__, "DISAPPEARED", member);
						return restart_election(member);
					} else {
						if (member_group_master(member, i64v)) {
							member_set_master_id(member, member->myid);
							member_set_status(member, STEP_PRELEAD);
							return defer_GETVERS(member);
						} else {
							member_set_master_id(member, g_array_index(i64v, gint64, 0));
							member_set_status(member, STEP_PRELOST);
							zrc = step_AskMaster_start(member);
							if (zrc != ZOK) {
								GRID_WARN("AskMaster failed [%s.%s] [%s] : (%d) %s",
										member->name.base, member->name.type, hashstr_str(member->key),
										zrc, zerror(zrc));
								return become_leaver(member);
							} else {
								return defer_GETVERS(member);
							}
						}
					}
					g_assert_not_reached ();
					return;
				case EVT_LIST_KO:
				case EVT_NODE_LEFT:
					return become_candidate(member);
				default:
					GRID_DEBUG("IGNORED");
					return;
			}
			return;

		case STEP_PRELOST:
			EXTRA_ASSERT(member->myid != -1);
			EXTRA_ASSERT(member->master_id != -1);
			EXTRA_ASSERT(member->master_id != member->myid);
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_RESYNC_REQ:
					return member_ask_RESYNC_if_not_pending(member);
				case EVT_DISCONNECTED:
					return;

				case EVT_CREATE_OK:
				case EVT_CREATE_KO:
				case EVT_LEAVE_OK:
				case EVT_LEAVE_KO:
				case EVT_LIST_OK:
				case EVT_LIST_KO:
					member_warn("ABNORMAL", member);
					return;

				case EVT_MASTER_KO:
				case EVT_MASTER_EMPTY:
					return become_leaver(member);

				case EVT_MASTER_CHANGE:
					member->delayed_MASTER_CHANGE = 1;
					return;

				case EVT_NODE_LEFT:
					member->delayed_NODE_LEFT = 1;
					return;

				case EVT_EXITING:
					return become_leaver(member);

				case EVT_MASTER_OK:
					EXTRA_ASSERT(evt_arg != NULL);
					member_set_master_url(member, (gchar*)evt_arg);
					return member_finish_PRELOST(member);

				case EVT_GETVERS_OK:
					if (member_concerned_by_GETVERS(member, evt_arg))
						member_finish_PRELOST(member);
					return;

				case EVT_GETVERS_OUTDATED:
					if (member_concerned_by_GETVERS(member, evt_arg)) {
						member_ask_RESYNC_if_not_pending(member);
						// No need to finish, the RESYNC itself is a
						// pending action avoiding the finish
					}
					return;

				case EVT_GETVERS_CONCURRENT:
					if (member_concerned_by_GETVERS(member, evt_arg)) {
						++ member->concurrent_GETVERS;
						member_finish_PRELOST(member);
					}
					return;

				case EVT_GETVERS_ERROR:
					if (member_concerned_by_GETVERS(member, evt_arg)) {
						++ member->errors_GETVERS;
						member_finish_PRELOST(member);
					}
					return;

				case EVT_RESYNC_DONE:
					if (member->master_url)
						member_finish_PRELOST(member);
					return;
			}
			return;

		case STEP_PRELEAD:
			EXTRA_ASSERT(member->myid != -1);
			EXTRA_ASSERT(member->master_id == member->myid);
			EXTRA_ASSERT(member->master_url == NULL);
			EXTRA_ASSERT(!BOOL(member->delayed_MASTER_CHANGE));
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_RESYNC_REQ:
				case EVT_EXITING:
					return become_leaver(member);
				case EVT_DISCONNECTED:
					return;

				case EVT_CREATE_OK:
				case EVT_CREATE_KO:
				case EVT_MASTER_OK:
				case EVT_MASTER_KO:
				case EVT_LIST_OK:
				case EVT_LIST_KO:
				case EVT_LEAVE_OK:
				case EVT_LEAVE_KO:
				case EVT_MASTER_EMPTY:
				case EVT_RESYNC_DONE:
				case EVT_MASTER_CHANGE:
					/* these operations shoudl not happen while in PRELEAD */
					member_warn("ABNORMAL", member);
					return;

				case EVT_NODE_LEFT:
					member->delayed_NODE_LEFT = 1;
					return;

				case EVT_GETVERS_OUTDATED:
					if (member_concerned_by_GETVERS(member, evt_arg))
						become_leaver(member);
					return;

				case EVT_GETVERS_OK:
					if (member_concerned_by_GETVERS(member, evt_arg))
						member_finish_PRELEAD(member);
					return;

				case EVT_GETVERS_CONCURRENT:
					if (member_concerned_by_GETVERS(member, evt_arg)) {
						++ member->concurrent_GETVERS;
						member_finish_PRELEAD(member);
					}
					return;

				case EVT_GETVERS_ERROR:
					if (member_concerned_by_GETVERS(member, evt_arg)) {
						++ member->errors_GETVERS;
						member_finish_PRELEAD(member);
					}
					return;
			}
			return;

		case STEP_LEAVING:
			switch (evt) {
				case EVT_NONE:
					if (!member->manager->exiting)
						member->requested_USE = 1;
					return;
				case EVT_RESYNC_REQ:
					return member_ask_RESYNC_if_not_pending(member);
				case EVT_LEAVE_OK:
					GRID_INFO("LEFT election [%s.%s]", member->name.base, member->name.type);
					member_reset(member);
					return member_set_status(member, STEP_NONE);
				case EVT_LEAVE_KO:
					return become_leaver(member);
				default:
					GRID_DEBUG("IGNORED");
					return;
			}
			return;

		case STEP_LOST:
			EXTRA_ASSERT(member->myid != -1);
			EXTRA_ASSERT(member->master_id != -1);
			EXTRA_ASSERT(member->master_id != member->myid);
			EXTRA_ASSERT(member->master_url != NULL);
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_EXITING:
					return become_leaver(member);
				case EVT_NODE_LEFT:
					member_warn("DELETED", member);
					return become_leaver(member);
				case EVT_MASTER_KO:
				case EVT_MASTER_EMPTY:
				case EVT_MASTER_CHANGE:
					return become_candidate(member);
				case EVT_RESYNC_REQ:
					member_ask_RESYNC_if_not_pending (member);
					return member_set_status(member, STEP_PRELOST);
				default:
					GRID_DEBUG("IGNORED");
					return;
			}
			return;

		case STEP_LEADER:
			EXTRA_ASSERT(member->myid != -1);
			EXTRA_ASSERT(member->master_id == member->myid);
			EXTRA_ASSERT(member->master_url == NULL);
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_RESYNC_REQ:
				case EVT_EXITING:
					return become_leaver(member);
				case EVT_NODE_LEFT:
					member_warn("DELETED", member);
					return become_leaver(member);
				default:
					GRID_DEBUG("IGNORED");
					return;
			}
			return;

		case STEP_FAILED:
			switch (evt) {
				case EVT_NONE:
					return;
				case EVT_RESYNC_REQ:
					member_ask_RESYNC_if_not_pending(member);
					return restart_election(member);
				case EVT_CREATE_KO:
				case EVT_NODE_LEFT:
					return member_reset(member);
				default:
					GRID_DEBUG("IGNORED");
					return;
			}
			return;
	}
}

static void
_member_play_timer (struct election_member_s *member,
		enum sqlx_action_e action)
{
	if (action != ACTION_NONE && action != ACTION_PING)
		GRID_DEBUG("%s -> %s!", hashstr_str(member->key), _action2str(action));

	switch (action) {
		case ACTION_NONE:
			return;
		case ACTION_PING:
			return (void) defer_USE (member);
		case ACTION_FAIL:
			return become_failed (member);
		case ACTION_RETRY:
			switch(member->step) {
			case STEP_CANDOK:
				become_candidate(member);
				return;
			case STEP_PRELEAD:
			case STEP_PRELOST:
				defer_GETVERS(member);
				member->last_status = oio_ext_monotonic_time();
				return;
			default:
				member_warn("ACTION_RETRY shouldn't be called from this state",
						member);
				EXTRA_ASSERT(FALSE);
				return;
			}
		case ACTION_RESTART:
			return restart_election (member);
		case ACTION_LEAVE:
			return become_leaver (member);
		case ACTION_EXPIRE:
			return;
	}
}

static void
transition(struct election_member_s *member, enum event_type_e evt,
		void *evt_arg)
{
	enum election_step_e pre = member->step;
	_member_react(member, evt, evt_arg);
	enum election_step_e post = member->step;
	if (evt != EVT_NONE || pre != post)
		member_log_event(member, pre, evt);

	if (member->requested_PIPEFROM && member->step == STEP_PRELOST &&
			member->master_url)
		defer_PIPEFROM(member);
	if (member->requested_USE && (member->step == STEP_NONE))
		restart_election(member);

	return _member_play_timer (member, _member_get_next_action (member));
}

static void
transition_error(struct election_member_s *member,
		enum event_type_e evt, enum ZOO_ERRORS zrc)
{
	EXTRA_ASSERT(zrc != ZOK);

	static const enum ZOO_ERRORS reason_DISCONNECT[] = {
		/* ZSYSTEMERROR */
		ZRUNTIMEINCONSISTENCY,
		ZDATAINCONSISTENCY,
		ZMARSHALLINGERROR,
		ZUNIMPLEMENTED,
		ZINVALIDSTATE,

		/* ZAPIERROR */
		ZSESSIONEXPIRED,
		ZAUTHFAILED,

		ZOK /* end beacon */
	};

	static const enum ZOO_ERRORS reason_FAIL[]  = {
		/* ZSYSTEMERROR */
		ZCONNECTIONLOSS,
		ZOPERATIONTIMEOUT,
		ZBADARGUMENTS,

		/* ZAPIERROR */
		ZINVALIDCALLBACK,
		ZINVALIDACL,
		ZCLOSING,
		ZNOTHING,
		ZSESSIONMOVED,

		ZOK /* end beacon */
	};

	/* Special cases of <ZAPIERROR error that are strong enough to trigger
	 * an immediate exit. The ZK connection will be reset and previous
	 * session-ID will be abandonned. It's time for a hard reset of the
	 * election */
	for (const enum ZOO_ERRORS *prc = reason_DISCONNECT; *prc != ZOK; ++prc) {
		if (zrc == *prc)
			return transition(member, EVT_DISCONNECTED, NULL);
	}

	/* Several error cases denote the action that was impossible. The event
	 * to be used depends on the action, and is stored in evt */
	for (const enum ZOO_ERRORS *prc = reason_FAIL; *prc != ZOK; ++prc) {
		if (zrc == *prc)
			return transition(member, evt, NULL);
	}

	/* here should only remain ZAPIERRORs that have been managed already */
	g_assert(zrc < ZAPIERROR);
	return transition(member, evt, NULL);
}

static GSList *
_DEQUE_extract (struct deque_beacon_s *beacon)
{
	GSList *out = NULL;
	for (struct election_member_s *m=beacon->front; m ;m=m->next)
		out = g_slist_prepend (out, m);
	return g_slist_reverse (out);
}

guint
election_manager_play_timers (struct election_manager_s *manager, guint max)
{
	static const int steps[] = {
		STEP_NONE,
		STEP_FAILED, STEP_LEADER, STEP_LOST,
		//STEP_LEAVING,
		STEP_PRELEAD, STEP_PRELOST, STEP_CANDOK, STEP_CANDREQ,
		-1 /* stops the iteration */
	};

	guint count = 0;
	char descr[512];
	_manager_lock(manager);
	for (const int *pi=steps; *pi >= 0 && (!max || count < max) ;++pi) {
		struct deque_beacon_s *beacon = manager->members_by_state + *pi;
		if (!beacon->front)
			continue;
		/* working on an item of the list might alterate the list, and even
		   move the item itself to the tail. so working with a temp. list
		   avoids loops and wrong game on pointers. */
		GSList *l0 = _DEQUE_extract (beacon);
		for (GSList *l=l0; l && (!max || count < max) ;l=l->next) {

			struct election_member_s *m = l->data;
			enum sqlx_action_e action = _member_get_next_action (m);
			if (GRID_TRACE_ENABLED()) {
				member_descr (m, descr, sizeof(descr));
				GRID_TRACE("action [%s] %s", _action2str(action), descr);
			}

			if (action == ACTION_EXPIRE) {
				if (m->refcount == 1) {
					count ++;
					_DEQUE_remove (m);
					g_tree_remove (manager->members_by_key, m->key);
					member_unref (m);
					member_destroy (m);
				}
			} else {
				if (action != ACTION_NONE)
					count ++;

				enum election_step_e pre = m->step;
				_member_play_timer (m, action);
				enum election_step_e post = m->step;
				if (pre == post) {
					_DEQUE_remove (m);
					_DEQUE_add (m);
				}

			}
		}
		g_slist_free (l0);
		l0 = NULL;
	}
	_manager_unlock(manager);

	return count;
}
