/*
OpenIO SDS sqliterepo
Copyright (C) 2014 Worldine, original work as part of Redcurrant
Copyright (C) 2015 OpenIO, modified as part of OpenIO Software Defined Storage

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 3.0 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.
*/

#ifndef OIO_SDS__sqliterepo__election_h
# define OIO_SDS__sqliterepo__election_h 1

# include <glib.h>

struct election_counts_s
{
	guint total;
	guint none;
	guint pending;
	guint failed;
	guint master;
	guint slave;
};

/* Hidden type */
struct sqlx_repository_s;
struct election_manager_s;
struct sqlx_sync_s;
struct sqlx_peering_s;
struct sqlx_name_s;

struct replication_config_s
{
	/** Tells the unique ID of the local service. */
	const char * (*get_local_url) (gpointer ctx);

	/** Locate the replication peers of the base identified by <n,t>. An error
	 * means the base cannot be replicated or managed. A base not managed
	 *  locally must return an error. A base locally managed but not replicated
	 *  must return NULL and fill result with a NULL pointer or an empty array.
	 *
	 * @param ctx the pointer registered in the configuration
	 * @param n the logical name of the base (not the physical path)
	 * @param result a placeholder for the array of peers.
	 * @return NULL if 'result'
	 */
	GError* (*get_peers) (gpointer ctx, const struct sqlx_name_s *n,
			gboolean nocache, gchar ***result);

	/** Encapsulate the query for the DB's version */
	GError* (*get_version) (gpointer ctx, const struct sqlx_name_s *n,
			GTree **result);

	gpointer ctx; /**< An arbitrary pointer reused in every hook. */

	enum election_mode_e {
		ELECTION_MODE_NONE = 0, /**< No replication */
		ELECTION_MODE_QUORUM,   /**< A master is found when a qualified majority
								 * of member is present */
		ELECTION_MODE_GROUP     /**< A master is found when the whole group
								 * agree */
	} mode; /**< Is replication activated */
};

struct election_manager_vtable_s
{
	/** Destroys an election_manager created by election_manager_create() */
	void (*clean) (struct election_manager_s *);


	/* is replication configured, and how */
	enum election_mode_e (*get_mode) (const struct election_manager_s *);

	/* return the ID ofthe local service */
	const char * (*get_local) (const struct election_manager_s *);

	/* who are the peers for the given base */
	GError* (*election_get_peers) (struct election_manager_s *manager,
			const struct sqlx_name_s *n, gboolean nocache, gchar ***peers);

	/** Prepare the internal memory for the election context, but without
	 * starting the election. Usefull to prepare. */
	GError* (*election_init) (struct election_manager_s *manager,
			const struct sqlx_name_s *n);

	/** Triggers the global election mechanism then returns without
	 * waiting for a final status. */
	GError* (*election_start) (struct election_manager_s *manager,
			const struct sqlx_name_s *n);

	GError* (*election_exit) (struct election_manager_s *manager,
			const struct sqlx_name_s *n);

	/** Triggers the global election mechanism then wait for a final status
	 * have been locally hit.  */
	enum election_status_e (*election_get_status) (struct election_manager_s *manager,
			const struct sqlx_name_s *n, gchar **master_url);

	GError* (*election_trigger_RESYNC) (struct election_manager_s *m,
			const struct sqlx_name_s *n);
};

struct abstract_election_manager_s
{
	struct election_manager_vtable_s *vtable;
};

/* ------------------------------------------------------------------------- */

#define election_manager_clean(m) \
	((struct abstract_election_manager_s*)m)->vtable->clean(m)

enum election_mode_e election_manager_get_mode (const struct election_manager_s *);

const char * election_manager_get_local (const struct election_manager_s *m);

GError* election_get_peers (struct election_manager_s *manager,
		const struct sqlx_name_s *n, gboolean nocache, gchar ***peers);

#define election_init(m,n) \
	((struct abstract_election_manager_s*)m)->vtable->election_init(m,n)

#define election_start(m,n) \
	((struct abstract_election_manager_s*)m)->vtable->election_start(m,n)

#define election_exit(m,n) \
	((struct abstract_election_manager_s*)m)->vtable->election_exit(m,n)

#define election_get_status(m,n,pmaster) \
	((struct abstract_election_manager_s*)m)->vtable->election_get_status(m,n,pmaster)

#define election_manager_trigger_RESYNC(m,n) \
	((struct abstract_election_manager_s*)m)->vtable->election_trigger_RESYNC(m,n)

/* wraps election_get_peers() */
GError * election_has_peers (struct election_manager_s *m,
		const struct sqlx_name_s *n, gboolean nocache, gboolean *ppresent);

/* Implementation-specific operations -------------------------------------- */

/* Creates the election_manager structure.  */
GError* election_manager_create (struct replication_config_s *config,
		struct election_manager_s **result);

struct election_counts_s election_manager_count (struct election_manager_s *m);

/* Perform the 'timer' action on one item of each status.
   This includes expiring the election, retrying, pinging peers, etc.
   Returns the number of items activated. */
guint election_manager_play_timers (struct election_manager_s *m, guint max);

/* Similar to the MANAGER_CHECK macro, but not stripped in Release mode,
 * and returns a boolean instead of asserting. */
gboolean election_manager_is_operational(struct election_manager_s *manager);

void election_manager_exit_all (struct election_manager_s *m,
		gint64 oldest, gboolean persist);

void election_manager_whatabout (struct election_manager_s *m,
		const struct sqlx_name_s *n, gchar *d, gsize ds);

void election_manager_set_sync (struct election_manager_s *manager,
		struct sqlx_sync_s *sync);

void election_manager_set_peering (struct election_manager_s *m,
		struct sqlx_peering_s *peering);

#endif /*OIO_SDS__sqliterepo__election_h*/
